import axiosInstance from '@/configs/service';
import {
    Book,
    CommentLike,
    CreateStudyGroup,
    EditStudyGroup,
    MemberData,
    PaginationResponse,
    Replies,
    StudyGroup,
    StudyGroupMember,
    StudyGroupMembers,
    StudyGroupPenaltySummary,
    StudyGroupTask,
    StudyGroupTaskComment,
} from '@/types';

// Context Provider
export const getStudyGroupById = (studyGroupId: string) => {
    return axiosInstance.get<StudyGroup>(`/staff/study-group/${studyGroupId}/id`);
};

export const getStudyGroupMember = (studyGroupId: string) => {
    return axiosInstance.get<StudyGroupMember>(`/staff/study-group/${studyGroupId}/member`);
};

export const getStudyGroupTaskById = (studyGroupTaskId: string) => {
    return axiosInstance.get<StudyGroupTask>(`/staff/study-group/${studyGroupTaskId}/task-by-id`);
};

///////////////////
// Study Group
///////////////////

// Get
export const getStudyGroupList = (query: StudyGroup) => {
    return axiosInstance.get<PaginationResponse<StudyGroup>>('/staff/study-group', { params: query });
};

export const getBooks = () => {
    return axiosInstance.get<Book[]>(`/staff/study-group/books`);
};

export const getStudyGroupPenalty = (studyGroupId: string) => {
    return axiosInstance.get(`/staff/study-group/${studyGroupId}/study-group-penalty`);
};

export const getAllPenaltiesByStudyGroup = (studyGroupId: string) => {
    return axiosInstance.get<StudyGroupPenaltySummary[]>(`/staff/study-group/${studyGroupId}/calculate-penalties`);
};

export const getMemberListByPagination = (query: any, studyGroupId: string) => {
    return axiosInstance.get<PaginationResponse<MemberData>>(`/staff/study-group/${studyGroupId}/member-list`, {
        params: query,
    });
};

export const getStudyGroupMembers = (studyGroupId: string, query: StudyGroupMembers) => {
    return axiosInstance.get<PaginationResponse<StudyGroupMembers>>(`/staff/study-group/${studyGroupId}/members`, { params: query });
};

export const getExportStudyGroupMemberList = (query: StudyGroupMembers, studyGroupId: string) => {
    return axiosInstance.get(`/staff/study-group/member-export/${studyGroupId}`, { params: query });
};

// Add
export const addGroup = (body: CreateStudyGroup) => {
    return axiosInstance.post<StudyGroup>(`/staff/study-group`, body);
};

export const addMember = (studyGroupId: string, memberId: string) => {
    return axiosInstance.post<StudyGroupMember>(`/staff/study-group/${studyGroupId}/member/${memberId}`);
};

// Update
export const updateStudyGroupById = (studyGroupId: string, body: EditStudyGroup) => {
    return axiosInstance.put<StudyGroup>(`/staff/study-group/${studyGroupId}`, body);
};

// Delete
export const deleteGroupById = (studyGroupId: string) => {
    return axiosInstance.delete<StudyGroup>(`/staff/study-group/${studyGroupId}`);
};

export const deleteStudyGroupMemberById = (studyGroupId: string, memberId: string) => {
    return axiosInstance.delete<StudyGroupMember>(`/staff/study-group/${studyGroupId}/member/${memberId}`);
};

///////////////////
// Study Group Task
///////////////////

// Get
export const getTaskMemberPenalty = (studyGroupTaskId: string) => {
    return axiosInstance.get<StudyGroupTask>(`/staff/study-group/${studyGroupTaskId}/task-member-penalty`);
};

export const getTaskPenalty = (studyGroupTaskId: string) => {
    return axiosInstance.get<StudyGroupTask>(`/staff/study-group/${studyGroupTaskId}/task-penalty`);
};

// Add
export const generateStudyGroupTask = (studyGroupId: string) => {
    return axiosInstance.post<StudyGroupTask>(`/staff/study-group/${studyGroupId}/generate-task`);
};

// Update
export const updateDescription = (studyGroupTaskId: string, body: string) => {
    return axiosInstance.put<StudyGroupTask>(`/staff/study-group/${studyGroupTaskId}/description`, { description: body });
};

///////////////////
// Comment
///////////////////

// Get
export const getCommentsByTaskId = (studyGroupTaskId: string) => {
    return axiosInstance.get<StudyGroupTaskComment>(`/staff/study-group/${studyGroupTaskId}/task-comments-id`);
};

export const getCommentsByGroupId = (studyGroupId: string) => {
    return axiosInstance.get<StudyGroupTaskComment[]>(`/staff/study-group/${studyGroupId}/group-comments`);
};

export const getStudyGroupTaskComments = (studyGroupTaskId: string, query: any) => {
    return axiosInstance.get<PaginationResponse<StudyGroupTaskComment>>(`/staff/study-group/${studyGroupTaskId}/task-comments`, { params: query });
};

export const getOwnTaskComment = (studyGroupTaskId: string) => {
    return axiosInstance.get<StudyGroupTaskComment>(`/staff/study-group/${studyGroupTaskId}/own-task-comment`);
};

// Delete
export const deleteCommentById = (commentId: string) => {
    return axiosInstance.delete<StudyGroupTaskComment>(`/staff/study-group/${commentId}/comment`);
};

// Update
export const likeComment = (studyGroupTaskCommentId: string, staffId: string, studyGroupTaskId: string) => {
    return axiosInstance.post(`/staff/study-group/${studyGroupTaskId}/like/${studyGroupTaskCommentId}`, { staffId });
};

export const updateReply = (studyGroupTaskId: string, reply: string, commentId?: string) => {
    return axiosInstance.put(`/staff/study-group/${studyGroupTaskId}/reply`, { reply, commentId });
};

export const getTaskCommentLikes = (studyGroupTaskId: string) => {
    return axiosInstance.get<CommentLike[]>(`/staff/study-group/${studyGroupTaskId}/like`);
};

export const getReplies = (studyGroupTaskId: string, commentId: string) => {
    return axiosInstance.get<Replies[]>(`/staff/study-group/${studyGroupTaskId}/reply/${commentId}`);
};
