import axiosInstance, { uploadInstance } from '@/configs/service';
import {
    Book,
    BookAddParams,
    BookQuery,
    BookUpdateParams,
    MediaFormValue,
    MediaResponse,
    PaginationParams,
    PaginationResponse,
    SortOrder,
} from '@/types';
import { mediaUrlGenerator } from '@/utils';

export const bookKeys = {
    all: ['book'],
    add: ['book', 'add'],
    id: (id: string) => [...bookKeys.all, id],
    pagination: (pagination: PaginationParams = { page: 1, pageSize: 10, sortField: 'createdAt', sortOrder: SortOrder.DESC }) => [
        ...bookKeys.all,
        'pagination',
        pagination,
    ],
};

export const getBookList = (query: BookQuery) => {
    return axiosInstance.get<PaginationResponse<Book>>(`/staff/book`, { params: query });
};

export const addBook = (body: BookAddParams) => {
    return axiosInstance.post(`/staff/book/add`, body);
};

export const getBookById = (id: string) => {
    return axiosInstance.get<Book>(`/staff/book/${id}`);
};

export const uploadBookFile = async (url: string, option: any) => {
    const formData = new FormData();
    formData.append('file', option.file);

    try {
        const response = await uploadInstance.post<MediaResponse>(url, formData);

        option.onSuccess({
            uid: response.data.id,
            name: response.data.name,
            url: mediaUrlGenerator(response.data.key),
            status: 'done',
        } as MediaFormValue);
    } catch (error) {
        option.onError(error);
    }
};

export const updateBookById = (id: string, body: BookUpdateParams) => {
    return axiosInstance.put<Book>(`/staff/book/${id}`, body);
};

export const updateBookStatus = (id: string) => {
    return axiosInstance.put(`/staff/book/${id}/status`);
};

export const deleteBookById = (id: string) => {
    return axiosInstance.delete(`/staff/book/${id}`);
};
