import { Dispatch, PropsWithChildren, SetStateAction, createContext, useContext, useState } from 'react';
import { UseQueryResult, useQuery } from '@tanstack/react-query';
import dayjs, { Dayjs } from 'dayjs';
import useModal from '@/hooks/useModal';
import { getBirthdaysByDate } from '@/services/birthday';
import { CalendarMode, BirthdayDateQuery } from '@/types';
import BirthdayListModal from '@/components/birthday/modals/BirthdayList.modal';
import ExportData from '@/components/birthday/ExportData';

interface CalendarProps {
    mode: CalendarMode;
    setMode: Dispatch<SetStateAction<CalendarMode>>;
    date: Dayjs;
    setDate: Dispatch<SetStateAction<Dayjs>>;
    birthdayListModalOpen: boolean;
    setBirthdayListModalOpen: Dispatch<SetStateAction<boolean>>;
    selectedDate?: Dayjs;
    birthdayListQuery: UseQueryResult<BirthdayDateQuery[]>;
}
interface BirthdayContextType {
    calendar: CalendarProps;
}

const BirthdayContext = createContext<BirthdayContextType | null>(null);

interface BirthdayProviderProps extends PropsWithChildren {}

export const BirthdayProvider: React.FC<BirthdayProviderProps> = ({ children }) => {
    const [calendarMode, setCalendarMode] = useState<CalendarMode>(CalendarMode.Month);
    const [selectedDate, setSelectedDate] = useState<Dayjs>(dayjs.tz());
    const [birthdayListModalOpen, setBirthdayListModalOpen] = useModal();

    const birthdayListQuery = useQuery({
        queryKey: ['birthday', 'date', selectedDate],
        queryFn: async () => {
            const dateQuery: BirthdayDateQuery = {
                id: '',
                fullName: '',
                dateOfBirth: selectedDate.startOf('day').endOf('day').set('year', dayjs.tz().year()).toISOString(),
                address: '',
                phoneNumber: '',
            };
            const data = await getBirthdaysByDate(dateQuery);

            return data.data;
        },
    });

    const exportBirthdayListToCsv = async (): Promise<{ data: any }> => {
        birthdayListQuery.refetch();

        if (!birthdayListQuery.data) return { data: [] };

        // Filter birthdays for the current month
        const currentMonthBirthdays = birthdayListQuery.data.filter(
            (birthday: any) => dayjs.tz(birthday.dateOfBirth).format('MMMM') === dayjs(selectedDate, 'month').format('MMMM'),
        );

        return { data: currentMonthBirthdays };
    };

    const currentMonth = selectedDate.format('MMMM');

    return (
        <BirthdayContext.Provider
            value={{
                calendar: {
                    mode: calendarMode,
                    setMode: setCalendarMode,
                    date: selectedDate,
                    setDate: setSelectedDate,
                    birthdayListModalOpen,
                    setBirthdayListModalOpen,
                    selectedDate,
                    birthdayListQuery,
                },
            }}
        >
            <div className="flex justify-end">
                <ExportData fileName={`birthday-${currentMonth}`} getExportData={exportBirthdayListToCsv} />
            </div>

            {children}
            <BirthdayListModal />
        </BirthdayContext.Provider>
    );
};

export const useBirthdayContext = () => {
    const context = useContext(BirthdayContext);

    if (!context) {
        throw new Error('useBirthdayContext must be used within BirthdayProvider');
    }

    return context;
};
