import usePagination from '@/hooks/usePagination';
import { AxiosErrorResponse, BasePageProps, StudyGroup } from '@/types';
import { authentication, conditionalReturn, errorMessageFormatter, localeLinkGenerator } from '@/utils';
import { Button, Form, MenuProps, Space, Table, TableColumnProps, Tag } from 'antd';
import { useTranslation } from 'next-i18next';
import { toast } from 'react-toastify';
import { ColumnsType } from 'antd/es/table';
import Layout from '@/components/layouts';
import { GetServerSideProps } from 'next';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import { useQuery } from '@tanstack/react-query';
import FilterDrawer from '@/components/study-group/modals/filter';
import { PlusCircleOutlined } from '@ant-design/icons';
import { useState } from 'react';
import AddGroup from '@/components/study-group/modals/AddGroup';
import Link from 'next/link';
import ActionDropdown from '@/components/ui/ActionDropdown';
import DeleteGroupAction from '@/components/study-group/DeleteGroupAction';
import EditGroup from '@/components/study-group/modals/EditGroup';
import { getStudyGroupList } from '@/services/study-group';
import { dateTimeTransformer, dateTransformer } from '@/utils/timeTransformer';

const StudyGroupPage: React.FC<BasePageProps> = ({ staff }) => {
    const { t } = useTranslation(['study-group', 'common', 'messages', 'layout']);
    const [pagination, setPagination, paginationOnChange] = usePagination();
    const [filterStudyGroupForm] = Form.useForm();
    const [isAddGroupModalOpen, setIsAddGroupModalOpen] = useState<boolean>(false);
    const [isEditGroupModalOpen, setIsEditGroupModalOpen] = useState<boolean>(false);
    const [studyGroupId, setStudyGroupId] = useState<string>('');

    // Query
    const studyGroupListQuery = useQuery({
        queryKey: ['study-group', 'pagination', pagination],
        queryFn: async () => {
            const filterValues = filterStudyGroupForm.getFieldsValue();

            const query = {
                ...filterValues,
                page: pagination.page,
                pageSize: pagination.pageSize,
                sortField: pagination.sortField,
                sortOrder: pagination.sortOrder,
            };

            const response = await getStudyGroupList(query);

            setPagination((prev) => {
                return {
                    ...prev,
                    page: response.data?.page,
                    total: response.data?.total,
                };
            });

            return response.data?.rows;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Functions
    const onResetFilterHandler = () => {
        filterStudyGroupForm.resetFields();
        studyGroupListQuery.refetch();
    };

    // Data
    const columns: ColumnsType<StudyGroup> = [
        {
            title: t('group-name'),
            dataIndex: 'name',
            key: 'name',
            render: (_: unknown, studyGroup) => {
                return (
                    <Link href={`/study-group/${studyGroup.id}`}>
                        <b>{studyGroup.name}</b>
                    </Link>
                );
            },
        },
        {
            title: t('book-name'),
            dataIndex: 'book',
            key: 'book',
            render: (_: unknown, studyGroup) => {
                return <Link href={`/book/${studyGroup.book.id}`}>{studyGroup.book?.name}</Link>;
            },
        },
        {
            title: t('members'),
            dataIndex: 'members',
            key: 'members',
            width: 120,
            render: (_: unknown, studyGroup: StudyGroup) => {
                return studyGroup._count?.studyGroupMembers;
            },
        },
        {
            title: t('start-date'),
            dataIndex: 'startDate',
            key: 'startDate',
            render: (_: unknown, studyGroup) => {
                // Format to malaysia timezone
                return dateTransformer(studyGroup.startDate);
            },
        },
        {
            title: t('end-date'),
            dataIndex: 'endDate',
            key: 'endDate',
            render: (_: unknown, studyGroup) => {
                return dateTransformer(studyGroup.endDate);
            },
        },
        {
            title: t('status'),
            dataIndex: 'status',
            key: 'status',
            width: 130,
            render: (_: unknown, studyGroup) => {
                let tagColor, tagText;

                switch (studyGroup?.status) {
                    case 'DRAFT':
                        tagColor = '#CCCCCC';
                        tagText = t('draft');
                        break;
                    case 'GENERATED':
                        tagColor = '#0074D9';
                        tagText = t('generated');
                        break;
                    case 'ONGOING':
                        tagColor = '#FFD700';
                        tagText = t('ongoing');
                        break;
                    case 'COMPLETED':
                        tagColor = '#2ECC40';
                        tagText = t('completed');
                        break;
                }

                return <Tag color={tagColor}>{tagText}</Tag>;
            },
        },
        {
            title: t('common:created-at'),
            dataIndex: 'createdAt',
            key: 'createdAt',
            width: 200,
            render: (_: unknown, studyGroup) => {
                return dateTimeTransformer(studyGroup.createdAt);
            },
        },
        {
            title: t('common:actions'),
            dataIndex: 'id',
            key: 'id',
            width: 100,
            align: 'center',
            render: (_: unknown, studyGroup) => {
                const items: MenuProps['items'] = [
                    {
                        label: <Link href={`/study-group/${studyGroup.id}`}>{t('common:view')}</Link>,
                        key: 'view',
                        className: 'text-center',
                    },
                    ...conditionalReturn(!!staff.role.STUDY_GROUP_UPDATE, [
                        {
                            label: (
                                <Button
                                    onClick={() => {
                                        setIsEditGroupModalOpen(true);
                                        setStudyGroupId(studyGroup.id);
                                    }}
                                    className="!text-black hover:!text-black !px-14 border-none"
                                    ghost
                                >
                                    {t('common:edit')}
                                </Button>
                            ),
                            key: 'update',
                            className: 'text-center !p-0',
                        },
                    ]),
                    ...conditionalReturn(!!staff.role.STUDY_GROUP_DELETE, [
                        {
                            label: <DeleteGroupAction studyGroup={studyGroup} query={studyGroupListQuery} />,
                            key: 'delete',
                            danger: true,
                            className: 'text-center !p-0',
                        },
                    ]),
                ];
                return (
                    <Space>
                        <ActionDropdown items={items} />
                    </Space>
                );
            },
        },
    ] as TableColumnProps<StudyGroup>[];

    const breadCrumbItems = [
        {
            label: t('layout:study-group'),
            path: '/study-group',
        },
    ];

    return (
        <Layout
            staff={staff}
            breadCrumbItems={breadCrumbItems}
            seoConfig={{
                title: t('study-group'),
            }}
            withBackground
            activeMenu={['study-group']}
        >
            <div className="flex justify-between mb-4 flex-col sm:flex-row">
                <div className="flex">
                    <FilterDrawer
                        filterStudyGroupForm={filterStudyGroupForm}
                        onSearch={() => studyGroupListQuery.refetch()}
                        onReset={onResetFilterHandler}
                        loading={studyGroupListQuery.isLoading}
                    />
                    <Button type="link" className="list-none" onClick={onResetFilterHandler}>
                        {t('common:reset-filter')}
                    </Button>
                </div>
                <div className="mt-2 sm:mt-0">
                    {staff.role.STUDY_GROUP_CREATE && (
                        <Button icon={<PlusCircleOutlined />} type="primary" onClick={() => setIsAddGroupModalOpen(true)}>
                            {t('add-group')}
                        </Button>
                    )}
                </div>
            </div>
            <div className="table_container">
                <Table
                    columns={columns}
                    dataSource={studyGroupListQuery.data}
                    loading={studyGroupListQuery.isLoading}
                    rowKey={(record) => record.id}
                    scroll={{ x: 1000 }}
                    onChange={paginationOnChange}
                    pagination={{
                        current: pagination.page,
                        pageSize: pagination.pageSize,
                        defaultPageSize: 1,
                        showSizeChanger: true,
                        pageSizeOptions: [10, 25, 50, 100],
                        showTotal: (total, range) => {
                            return t('common:pagination', { range0: range[0], range1: range[1], total: total });
                        },
                        total: pagination.total,
                    }}
                />
            </div>
            <AddGroup open={isAddGroupModalOpen} setOpen={setIsAddGroupModalOpen} refetch={() => studyGroupListQuery.refetch()} />
            {isEditGroupModalOpen && (
                <EditGroup
                    open={isEditGroupModalOpen}
                    setOpen={setIsEditGroupModalOpen}
                    refetch={() => studyGroupListQuery.refetch()}
                    studyGroupId={studyGroupId}
                />
            )}
        </Layout>
    );
};

export default StudyGroupPage;

export const getServerSideProps: GetServerSideProps = async ({ locale, req, resolvedUrl }) => {
    try {
        const authResponse = await authentication(req, 'STUDY_GROUP_VIEW');

        return {
            props: {
                staff: authResponse,
                ...(await serverSideTranslations(locale as string, [
                    'study-group',
                    'study-group-task',
                    'layout',
                    'common',
                    'messages',
                    'auth',
                    'api-messages',
                ])),
            },
        };
    } catch (error: any) {
        if (error.response?.data?.unauthorized) {
            return {
                redirect: {
                    destination: localeLinkGenerator(locale, `/unauthorized`),
                    permanent: false,
                },
            };
        }

        return {
            redirect: {
                destination: localeLinkGenerator(locale, `/?redirect=/${resolvedUrl}`),
                permanent: false,
            },
        };
    }
};
