import Layout from '@/components/layouts';
import usePagination from '@/hooks/usePagination';
import { createStaff, deleteStaff, getStaffListByPagination, updateStaffStatus } from '@/services/staff';
import { AxiosErrorResponse, BasePageProps, CreateStaffParams, Staff, StaffStatus, TokenType } from '@/types';
import { authentication } from '@/utils/authentication';
import { errorMessageFormatter } from '@/utils/errorFormatter';
import localeLinkGenerator from '@/utils/localeLinkGenerator';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { Button, Form, MenuProps, Space, Table, TableColumnProps } from 'antd';
import { GetServerSideProps, NextPage } from 'next';
import { Trans, useTranslation } from 'next-i18next';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import Link from 'next/link';
import { useState } from 'react';
import { toast } from 'react-toastify';
import StaffStatusTag from '@/components/Status';
import conditionalReturn from '@/utils/conditionalReturn';
import dayjs from 'dayjs';
import AddStaffModal from '@/components/staff/modals/AddStaff';
import ConfirmationModal from '@/components/modals/ConfirmationModal';
import ActionDropdown from '@/components/ui/ActionDropdown';
import FilterDrawer from '@/components/staff/modals/Filter';
import ColumnSelector from '@/components/modals/ColumnSelector';
import { resendVerificationEmail } from '@/services/auth';
import { PlusCircleOutlined } from '@ant-design/icons';
import { dateTimeTransformer } from '@/utils/timeTransformer';

const StaffPage: NextPage<BasePageProps> = ({ staff }) => {
    const { t } = useTranslation(['staff', 'common', 'auth']);
    const queryClient = useQueryClient();
    const [pagination, setPagination, paginationOnChange] = usePagination<Staff>();
    const [selectedColumn, setSelectedColumn] = useState<string[]>(['fullName', 'email', 'status', 'lastActiveAt']);
    const [isAddStaffModalOpen, setIsAddStaffModalOpen] = useState<boolean>(false);

    // Form Instances
    const [filterStaffForm] = Form.useForm();
    const [addStaffForm] = Form.useForm();

    // Query
    const staffQuery = useQuery({
        queryKey: ['staff', 'pagination', pagination],
        queryFn: async () => {
            // Get the form values
            let searchedValue = filterStaffForm.getFieldsValue();

            const query = {
                ...searchedValue,
                page: pagination.page,
                pageSize: pagination.pageSize,
                sortField: pagination.sortField,
                sortOrder: pagination.sortOrder,
            };

            const res = await getStaffListByPagination(query);

            setPagination((prevValue) => {
                return {
                    ...prevValue,
                    page: res.data?.page,
                    total: res.data?.total,
                };
            });

            return res.data?.rows;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Mutations
    const createStaffMutation = useMutation({
        mutationFn: async (staff: CreateStaffParams) => {
            const res = await createStaff(staff);
            return res.data?.rows;
        },
        onSuccess: () => {
            setIsAddStaffModalOpen(false);
            addStaffForm.resetFields();
            staffQuery.refetch();
            queryClient.invalidateQueries(['staff'], { exact: true });
        },
    });

    const resendVerificationEmailMutation = useMutation({
        mutationFn: async (staffId: string) => {
            const res = await resendVerificationEmail(staffId);
            return res.data?.rows;
        },
        onSuccess: () => {
            staffQuery.refetch();
            queryClient.invalidateQueries(['staff'], { exact: true });
        },
    });

    const updateStaffStatusMutation = useMutation({
        mutationFn: async (staffId: string) => {
            const res = await updateStaffStatus(staffId);
            return res.data?.rows;
        },
        onSuccess: () => {
            staffQuery.refetch();
        },
    });

    const deleteStaffMutation = useMutation({
        mutationFn: async (staffId: string) => {
            const res = await deleteStaff(staffId);
            return res.data?.rows;
        },
        onSuccess: () => {
            staffQuery.refetch();
        },
    });

    // Functions
    const onCreateStaffHandler = () => {
        addStaffForm.validateFields().then((values: CreateStaffParams) => {
            toast.promise(createStaffMutation.mutateAsync(values), {
                pending: t('messages:creating-staff'),
                success: t('messages:staff-created'),
                error: {
                    render({ data }: any) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    const onResendVerificationEmailHandler = (email: string) => {
        toast.promise(resendVerificationEmailMutation.mutateAsync(email), {
            pending: t('messages:resending-email-verification'),
            success: t('messages:email-verification-resend'),
            error: {
                render({ data }: any) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    const onUpdateStaffStatusHandler = (staffId: string) => {
        toast.promise(updateStaffStatusMutation.mutateAsync(staffId), {
            pending: t('messages:updating-staff-status'),
            success: t('messages:staff-status-updated'),
            error: {
                render({ data }: any) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    const onDeleteStaffHandler = (id: string) => {
        toast.promise(deleteStaffMutation.mutateAsync(id), {
            pending: t('messages:deleting-staff'),
            success: t('messages:staff-deleted'),
            error: {
                render({ data }: any) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    const onResetHandler = () => {
        filterStaffForm.resetFields();
        staffQuery.refetch();
    };

    // Data
    const breadCrumbItems = [
        {
            label: t('staff'),
            path: '/',
        },
    ];

    const seoConfig = {
        title: t('staff'),
    };

    const columnOptions = [
        {
            label: t('full-name'),
            value: 'fullName',
        },
        {
            label: t('email'),
            value: 'email',
        },
        {
            label: t('status'),
            value: 'status',
        },
        {
            label: t('phone-number'),
            value: 'phoneNumber',
        },
        {
            label: t('last-active'),
            value: 'lastActiveAt',
        },
        {
            label: t('common:created-at'),
            value: 'createdAt',
        },
    ];

    const columns = [
        // Name
        {
            dataIndex: 'id',
            title: t('full-name'),
            sorter: true,
            render: (_: unknown, staff: Staff) => {
                return (
                    <Link className="font-bold" href={`/staff/${staff.id}`}>
                        {staff.fullName}
                    </Link>
                );
            },
        },
        // Email
        {
            dataIndex: 'email',
            title: t('email'),
            sorter: true,
        },
        // Status
        {
            dataIndex: 'status',
            title: t('status'),
            sorter: true,
            render: (_: unknown, staff: Staff) => {
                return <StaffStatusTag user={staff} />;
            },
        },
        // Phone Number
        ...conditionalReturn(selectedColumn.includes('phoneNumber'), [
            {
                title: t('phone-number'),
                dataIndex: 'phoneNumber',
                sorter: true,
            },
        ]),
        // Last Active
        ...conditionalReturn(selectedColumn.includes('lastActiveAt'), [
            {
                title: t('last-active'),
                dataIndex: 'lastActiveAt',
                width: 150,
                sorter: true,
                render: (lastActive: string) => {
                    return lastActive !== null ? dateTimeTransformer(lastActive) : '';
                },
            },
        ]),
        // Created At
        ...conditionalReturn(selectedColumn.includes('createdAt'), [
            {
                title: t('common:created-at'),
                dataIndex: 'createdAt',
                width: 150,
                sorter: true,
                render: (createdAt: string) => {
                    return createdAt !== null ? dateTimeTransformer(createdAt) : '';
                },
            },
        ]),
        // Actions
        {
            dataIndex: 'id',
            width: 64,
            fixed: 'right',
            render: (_: unknown, record: Staff) => {
                const action = record.status === StaffStatus.ACTIVE ? t('common:deactivate') : t('common:activate');
                const verificationToken = record.tokens.filter((token) => token.type === TokenType.CONFIRMATION);
                const items: MenuProps['items'] = [
                    // View Staff Details
                    {
                        label: <Link href={`/staff/${record.id}`}>{t('common:view')}</Link>,
                        key: 'view',
                        className: 'text-center',
                    },
                    // Resend Email Verification
                    ...conditionalReturn(
                        !record.password && (verificationToken.length === 0 || dayjs.tz(verificationToken[0]?.expiredAt).isBefore(dayjs.tz())),
                        [
                            {
                                label: (
                                    <ConfirmationModal
                                        title={t('common:resend-email-verification')}
                                        message={
                                            <Trans
                                                i18nKey={'messages:are-you-sure-you-want-to-resend-email-verification-to'}
                                                components={{ strong: <strong /> }}
                                                values={{ name: record.fullName }}
                                            />
                                        }
                                        okText={t('common:resend')}
                                        onOk={() => {
                                            onResendVerificationEmailHandler(record.email);
                                        }}
                                        reason={false}
                                        width={400}
                                    >
                                        <Button
                                            loading={resendVerificationEmailMutation.isLoading}
                                            type="text"
                                            block
                                            className="!p-0"
                                            style={{ transition: 'none' }}
                                        >
                                            {t('common:resend-email')}
                                        </Button>
                                    </ConfirmationModal>
                                ),
                                key: 'resend',
                                className: '!p-0',
                            },
                        ],
                    ),
                    // Activate / Deactivate Staff
                    ...conditionalReturn(staff.role.STAFF_UPDATE, [
                        {
                            label: (
                                <ConfirmationModal
                                    title={action}
                                    message={
                                        <Trans
                                            i18nKey={'messages:are-you-sure-you-want-to-activate-or-deactivate'}
                                            components={{ strong: <strong /> }}
                                            values={{ name: record.fullName, action: action.toLowerCase() }}
                                        />
                                    }
                                    okText={t(action)}
                                    onOk={() => {
                                        onUpdateStaffStatusHandler(record.id);
                                    }}
                                    reason={false}
                                    okButtonProps={{
                                        danger: true,
                                    }}
                                    width={400}
                                >
                                    <Button
                                        loading={updateStaffStatusMutation.isLoading}
                                        type="text"
                                        block
                                        className="!p-0"
                                        style={{ transition: 'none' }}
                                    >
                                        {t(action)}
                                    </Button>
                                </ConfirmationModal>
                            ),
                            key: 'activate',
                            className: '!p-0',
                        },
                    ]),
                    // Delete Staff
                    ...conditionalReturn(staff.role.STAFF_DELETE, [
                        {
                            label: (
                                <ConfirmationModal
                                    title={t('common:delete-confirmation')}
                                    message={
                                        <Trans
                                            i18nKey={'messages:are-you-sure-you-want-to-delete'}
                                            components={{ strong: <strong /> }}
                                            values={{ name: record.fullName }}
                                        />
                                    }
                                    okText={t('common:delete')}
                                    onOk={() => {
                                        onDeleteStaffHandler(record.id);
                                    }}
                                    reason={false}
                                    okButtonProps={{
                                        danger: true,
                                    }}
                                    width={400}
                                >
                                    <Button
                                        loading={deleteStaffMutation.isLoading}
                                        type="text"
                                        block
                                        className="!p-0 hover:!text-white"
                                        style={{ transition: 'none' }}
                                    >
                                        {t('common:delete')}
                                    </Button>
                                </ConfirmationModal>
                            ),
                            key: 'delete',
                            danger: true,
                            className: '!p-0',
                        },
                    ]),
                ];

                return (
                    <Space>
                        <ActionDropdown items={items} />
                    </Space>
                );
            },
        },
    ] as TableColumnProps<Staff>[];

    return (
        <Layout
            breadCrumbItems={breadCrumbItems}
            activeMenu={['staff']}
            seoConfig={seoConfig}
            withBackground
            staff={staff}
            activeDropdown={['users']}
        >
            <div className="flex justify-between mb-4 flex-col sm:flex-row">
                {/* Header */}
                <div className="flex">
                    {/* Filter Drawer */}

                    <FilterDrawer
                        filterStaffForm={filterStaffForm}
                        onReset={onResetHandler}
                        onSearch={() => {
                            staffQuery.refetch();
                        }}
                        loading={staffQuery.isLoading}
                    />

                    {/* Reset Filter Button */}

                    <Button type="link" className="list-none" onClick={onResetHandler}>
                        {t('common:reset-filter')}
                    </Button>
                </div>
                {/* Add Staff Button */}
                <div className="mt-2 sm:mt-0">
                    {staff.role.STAFF_CREATE && (
                        <Button type="primary" icon={<PlusCircleOutlined />} onClick={() => setIsAddStaffModalOpen(true)}>
                            {t('add-staff')}
                        </Button>
                    )}
                </div>
            </div>
            <div className="table_container">
                {/* Column Selector */}
                <div className="flex justify-end config_container mb-4">
                    <ColumnSelector options={columnOptions} column={selectedColumn} setColumn={setSelectedColumn} />
                </div>
                {/* Table */}
                <Table
                    columns={columns}
                    dataSource={staffQuery.data}
                    loading={staffQuery.isLoading}
                    rowKey={(record) => record.id}
                    scroll={{ x: 1000 }}
                    onChange={paginationOnChange}
                    pagination={{
                        current: pagination.page,
                        pageSize: pagination.pageSize,
                        defaultPageSize: 1,
                        showSizeChanger: true,
                        pageSizeOptions: [10, 25, 50, 100],
                        showTotal: (total, range) => t('common:pagination', { range0: range[0], range1: range[1], total: total }),
                        total: pagination.total,
                    }}
                />
            </div>
            {/* Add Staff Modal */}
            <AddStaffModal
                isLoaded={createStaffMutation.isLoading}
                form={addStaffForm}
                open={isAddStaffModalOpen}
                setOpen={setIsAddStaffModalOpen}
                onCreate={onCreateStaffHandler}
            />
        </Layout>
    );
};

export default StaffPage;

export const getServerSideProps: GetServerSideProps = async ({ locale, req, resolvedUrl }) => {
    try {
        const authResponse = await authentication(req, 'STAFF_VIEW');

        return {
            props: {
                staff: authResponse,
                ...(await serverSideTranslations(locale as string, ['staff', 'layout', 'common', 'messages', 'auth', 'api-messages'])),
            },
        };
    } catch (error: any) {
        if (error.response?.data?.unauthorized) {
            return {
                redirect: {
                    destination: localeLinkGenerator(locale, `/unauthorized`),
                    permanent: false,
                },
            };
        }

        return {
            redirect: {
                destination: localeLinkGenerator(locale, `/?redirect=/${resolvedUrl}`),
                permanent: false,
            },
        };
    }
};
