import { toast } from 'react-toastify';
import { GetServerSideProps, NextPage } from 'next';
import { useTranslation } from 'next-i18next';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import { useQuery, useQueryClient } from '@tanstack/react-query';
import { Tabs } from 'antd';
import Credential from '@/components/my-profile/tabs/Credential';
import Profile from '@/components/my-profile/tabs/Profile';
import Layout from '@/components/layouts';
import { getMyProfile } from '@/services/my-profile';
import { AxiosErrorResponse, BasePageProps } from '@/types';
import { authentication } from '@/utils/authentication';
import { errorMessageFormatter } from '@/utils/errorFormatter';
import localeLinkGenerator from '@/utils/localeLinkGenerator';
import { conditionalReturn } from '@/utils';
import ActivityLog from '@/components/activityLog';

const MyProfile: NextPage<BasePageProps> = ({ staff }) => {
    const { t } = useTranslation(['my-profile', 'layout', 'common', 'messages']);
    const staffId = staff.id;
    const queryClient = useQueryClient();

    const breadCrumbItems = [
        {
            label: t('my-profile'),
            path: '/my-profile',
        },
    ];

    const staffQuery = useQuery({
        queryKey: ['staff', staffId],
        keepPreviousData: true,
        queryFn: async () => {
            const res = await getMyProfile();
            return res.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    const onTabChangeHandler = (key: string) => {
        if (key === 'activityLog') {
            queryClient.invalidateQueries(['activityLog']);
        }
    };

    const tabItems = [
        // Profile Tab
        {
            label: t('common:profile'),
            key: 'profile',
            children: <Profile staffQuery={staffQuery} />,
        },
        // Credential Tab
        {
            label: t('common:credential'),
            key: 'credential',
            children: <Credential staffId={staffId} staffQuery={staffQuery} />,
        },
        ...conditionalReturn(staff.role.ACTIVITY_LOG, [
            {
                key: 'activityLog',
                label: t('common:activity-log'),
                children: <ActivityLog target={`my-profile:${staffId}`} />,
            },
        ]),
    ];

    return (
        <Layout
            staff={staff}
            breadCrumbItems={breadCrumbItems}
            seoConfig={{
                title: t('my-profile'),
            }}
            withBackground
            activeMenu={['my-profile']}
        >
            <Tabs items={tabItems} onChange={onTabChangeHandler} />
        </Layout>
    );
};

export default MyProfile;

export const getServerSideProps: GetServerSideProps = async ({ locale, req, resolvedUrl }) => {
    try {
        const authResponse = await authentication(req);

        return {
            props: {
                staff: authResponse,
                ...(await serverSideTranslations(locale as string, ['my-profile', 'layout', 'common', 'messages', 'auth', 'api-messages'])),
            },
        };
    } catch (error: any) {
        if (error.response?.data?.unauthorized) {
            return {
                redirect: {
                    destination: localeLinkGenerator(locale, `/unauthorized`),
                    permanent: false,
                },
            };
        }

        return {
            redirect: {
                destination: localeLinkGenerator(locale, `/?redirect=/${resolvedUrl}`),
                permanent: false,
            },
        };
    }
};
