import { GetServerSideProps, NextPage } from 'next';
import { useTranslation } from 'next-i18next';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import Layout from '@/components/layouts';
import { BasePageProps } from '@/types';
import { authentication } from '@/utils/authentication';
import localeLinkGenerator from '@/utils/localeLinkGenerator';
import { Button, Form, Skeleton } from 'antd';
import { ColumnsType } from 'antd/es/table';
import { MenuProps, Table } from 'antd/lib';
import ActionDropdown from '@/components/ui/ActionDropdown';
import usePagination from '@/hooks/usePagination';
import { useQuery } from '@tanstack/react-query';
import BookOrderStatusTag from '@/components/book-order/BookOrderStatusTag';
import { useState } from 'react';
import FilterDrawer from '@/components/book-order/modals/Filter';
import { getBookOrderList } from '@/services/book-order';
import { BookOrderList, BookOrderStatus } from '@/types/book-order';
import Link from 'next/link';
import { EyeOutlined } from '@ant-design/icons';
import ViewBookOrderDetails from '@/components/book-order/modals/ViewBookOrderDetails';
import { conditionalReturn, currencyFormatter } from '@/utils';
import DeleteBookOrderAction from '@/components/book-order/DeleteBookOrderAction';
import ExportData from '@/components/book-order/ExportData';
import { dateTimeTransformer } from '@/utils/timeTransformer';

const Index: NextPage<BasePageProps> = ({ staff }) => {
    const { t } = useTranslation(['book-order', 'layout', 'common', 'messages']);
    const [pagination, setPagination, paginationOnChange] = usePagination<BookOrderList>();
    const [filterBookOrderRequestForm] = Form.useForm();

    const [selectedBookOrderId, setSelectedBookOrderId] = useState<string | null>(null);
    const [isViewBookOrderModalOpen, setIsViewBookOrderModalOpen] = useState<boolean>(false);

    // Query
    const {
        isFetching: bookOrderIsFetching,
        data: bookOrderData,
        refetch: bookOrderRefetch,
    } = useQuery({
        queryKey: ['book-order', pagination],
        queryFn: async () => {
            const filterValues = filterBookOrderRequestForm.getFieldsValue();

            const query = {
                ...filterValues,
                page: pagination.page,
                pageSize: pagination.pageSize,
                sortField: pagination.sortField,
                sortOrder: pagination.sortOrder,
            };

            const response = await getBookOrderList(query);

            setPagination((prev) => {
                return {
                    ...prev,
                    page: response.data.page,
                    total: response.data.total,
                };
            });

            return response.data.rows;
        },
    });

    const columns: ColumnsType<BookOrderList> = [
        {
            title: t('ref-no'),
            dataIndex: 'displayId',
            key: 'displayId',
            width: 160,
            render: (displayId: string, record) => {
                return (
                    <Button type="link" onClick={() => onViewBookOrderHandler(record.id)} className="p-0 font-bold">
                        {displayId}
                    </Button>
                );
            },
        },
        {
            title: t('member-name'),
            dataIndex: 'id',
            key: 'member-name',
            render: (id: string, record) => {
                return (
                    <Link href={`/member/${record.member.id}`} ref={null}>
                        <Button type="link" className="p-0 font-bold">
                            {record.member.fullName}
                        </Button>
                    </Link>
                );
            },
        },
        {
            title: t('book-name'),
            dataIndex: 'id',
            key: 'book-name',
            render: (id: string, record) => {
                return (
                    <Link href={`/book/${record.book.id}`} ref={null}>
                        <Button type="link" className="p-0 font-bold">
                            {record.book.name}
                        </Button>
                    </Link>
                );
            },
        },
        {
            title: t('book-quantity'),
            dataIndex: 'quantity',
            key: 'quantity',
        },
        {
            title: t('book-price'),
            dataIndex: 'bookPrice',
            key: 'book-price',
            width: 200,
            render: (bookPrice: number) => {
                return `MYR ${currencyFormatter(bookPrice)}`;
            },
        },
        {
            title: t('total-amount'),
            dataIndex: 'bookPrice',
            key: 'total-amount',
            width: 220,
            render: (bookPrice: number, record) => {
                return `MYR ${currencyFormatter(bookPrice * record.quantity)}`;
            },
        },
        {
            title: t('payment-status'),
            dataIndex: 'status',
            key: 'status',
            width: 160,
            render: (status: BookOrderStatus) => <BookOrderStatusTag status={status} />,
        },
        {
            title: t('common:created-at'),
            dataIndex: 'createdAt',
            key: 'createdAt',
            width: 180,
            render: (createdAt: string) => {
                return dateTimeTransformer(createdAt);
            },
        },
        {
            title: t('invoice'),
            dataIndex: 'id',
            key: 'invoice',
            width: 80,
            render: (bookOrderId: string) => {
                return (
                    <Link href={`/book-order-bill/invoice/${bookOrderId}`} target="_blank" className="flex justify-center">
                        <EyeOutlined className="text-base" />
                    </Link>
                );
            },
        },
        {
            title: t('receipt'),
            dataIndex: 'id',
            key: 'receipt',
            width: 80,
            render: (bookOrderId: string, bookOrder) => {
                return (
                    <div className="flex justify-center">
                        {bookOrder.status === BookOrderStatus.PAID ? (
                            <Link href={`/book-order-bill/receipt/${bookOrderId}`} target="_blank" className="flex justify-center">
                                <EyeOutlined className="text-base" />
                            </Link>
                        ) : (
                            '-'
                        )}
                    </div>
                );
            },
        },
        {
            title: t('common:actions'),
            dataIndex: 'createdAt',
            key: 'action',
            width: 80,
            render: (_: unknown, record) => {
                const items: MenuProps['items'] = [
                    {
                        label: t('common:view'),
                        key: 'view',
                        onClick: () => onViewBookOrderHandler(record.id),
                        className: 'text-center',
                    },
                    ...conditionalReturn(!!staff.role.STUDY_GROUP_DELETE, [
                        {
                            label: <DeleteBookOrderAction bookOrder={record} refetch={bookOrderRefetch} />,
                            key: 'delete',
                            danger: true,
                            className: 'text-center !p-0',
                        },
                    ]),
                ];

                return <ActionDropdown items={items} />;
            },
        },
    ];

    // Functions
    const exportBookOrderToCsv = async (): Promise<{ data: any }> => {
        bookOrderRefetch();

        if (!bookOrderData) return { data: [] };

        const csvDataTemplate = bookOrderData.map((bookOrder: any) => {
            return {
                refNo: bookOrder.displayId,
                memberName: bookOrder.member.fullName,
                memberAddress: bookOrder.member.address,
                memberPhone: bookOrder.member.phoneNumber,
                bookName: bookOrder.book.name,
                bookPrice: bookOrder.bookPrice,
                bookQuantity: bookOrder.quantity,
                paymentStatus: bookOrder.status,
                createdAt: dateTimeTransformer(bookOrder.createdAt),
            };
        });

        return { data: csvDataTemplate };
    };

    const onViewBookOrderHandler = (bookOrderId: string) => {
        setSelectedBookOrderId(bookOrderId);
        setIsViewBookOrderModalOpen(true);
    };

    const onResetFilterHandler = () => {
        filterBookOrderRequestForm.resetFields();
        bookOrderRefetch();
    };

    const seoConfig = {
        title: t('book-order-history'),
    };

    const breadCrumbItems = [
        {
            label: t('layout:book-order-history'),
            path: '/',
        },
    ];

    return (
        <Layout staff={staff} breadCrumbItems={breadCrumbItems} activeMenu={['book-order-history']} seoConfig={seoConfig} withBackground>
            <div className="flex flex-col sm:flex-row justify-between mb-4">
                <div className="flex mb-2 sm:mb-0">
                    <FilterDrawer
                        filterBookOrderRequestForm={filterBookOrderRequestForm}
                        onSearch={bookOrderRefetch}
                        onReset={onResetFilterHandler}
                        loading={bookOrderIsFetching}
                    />
                    <Button type="link" className="list-none" onClick={onResetFilterHandler}>
                        {t('common:reset-filter')}
                    </Button>
                </div>
                <div className="flex flex-col sm:flex-row sm:gap-4">
                    <div className="flex flex-row sm:justify-start">
                        <ExportData getExportData={exportBookOrderToCsv} />
                    </div>
                    <div className="flex flex-row sm:justify-start mt-2 sm:mt-0">
                        <Link href="/book-order-bill/quotation" target="_blank">
                            <Button type="primary" ghost>
                                {t('generate-quotation')}
                            </Button>
                        </Link>
                    </div>
                </div>
            </div>
            {bookOrderIsFetching ? (
                <Skeleton active />
            ) : (
                <>
                    <Table
                        columns={columns}
                        dataSource={bookOrderData}
                        loading={bookOrderIsFetching}
                        rowKey={(record) => record.id}
                        scroll={{ x: 1000 }}
                        onChange={paginationOnChange}
                        pagination={{
                            current: pagination.page,
                            pageSize: pagination.pageSize,
                            defaultPageSize: 1,
                            showSizeChanger: true,
                            pageSizeOptions: [10, 25, 50, 100],
                            showTotal: (total, range) => t('common:pagination', { range0: range[0], range1: range[1], total: total }),
                            total: pagination.total,
                        }}
                    />
                    <ViewBookOrderDetails
                        open={isViewBookOrderModalOpen}
                        setOpen={setIsViewBookOrderModalOpen}
                        selectedBookOrderId={selectedBookOrderId}
                    />
                </>
            )}
        </Layout>
    );
};

export const getServerSideProps: GetServerSideProps = async ({ req, locale, resolvedUrl }) => {
    try {
        const authResponse = await authentication(req, 'BOOK_ORDER_VIEW');

        return {
            props: {
                staff: authResponse,
                ...(await serverSideTranslations(locale as string, ['book-order', 'layout', 'common', 'messages', 'api-messages'])),
            },
        };
    } catch (error: any) {
        if (error.response?.data?.unauthorized) {
            return {
                redirect: {
                    destination: localeLinkGenerator(locale, `/unauthorized`),
                    permanent: false,
                },
            };
        }

        return {
            redirect: {
                destination: localeLinkGenerator(locale, `/?redirect=/${resolvedUrl}`),
                permanent: false,
            },
        };
    }
};

export default Index;
