import { AxiosErrorResponse, Penalty, StudyGroupStatus, StudyGroupTask } from '@/types';
import { useTranslation } from 'next-i18next';
import { Button, Card, Divider, Empty, Tooltip } from 'antd';
import { PlusCircleOutlined } from '@ant-design/icons';
import { useStaffContext } from '@/providers/StaffContext';
import React from 'react';
import { AiOutlineExclamationCircle } from 'react-icons/ai';
import { useStudyGroupContext } from '@/providers/StudyGroupContext';
import { currencyFormatter, errorMessageFormatter } from '@/utils';
import { useMutation, useQuery } from '@tanstack/react-query';
import ConfirmationModal from '@/components/modals/ConfirmationModal';
import { generateStudyGroupTask, getCommentsByGroupId } from '@/services/study-group';
import PreviewTask from '@/components/study-group-task/modals/PreviewTask';
import ExportData from '../ExportData';
import { useRouter } from 'next/router';
import { toast } from 'react-toastify';
import { dateTransformer } from '@/utils/timeTransformer';

const Task: React.FC = () => {
    const { t } = useTranslation(['study-group', 'common', 'messages', 'layout']);
    const { staff } = useStaffContext();
    const { studyGroup, setStudyGroupTaskId } = useStudyGroupContext();
    const [isPreviewTaskModal, setIsPreviewTaskModal] = React.useState<boolean>(false);
    const router = useRouter();
    const { studyGroupId } = router.query;

    const studyGroupData = studyGroup.data;
    const studyGroupTaskData = studyGroup.data?.studyGroupTask;

    const studyGroupComments = useQuery({
        queryKey: ['group-comments', studyGroupId],
        queryFn: async () => {
            const response = await getCommentsByGroupId(studyGroupId as string);
            return response.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Mutation
    const generateTaskMutation = useMutation({
        mutationFn: async () => {
            const response = await generateStudyGroupTask(studyGroupData?.id as string);
            return response;
        },
        onSuccess: () => {
            studyGroup.refetch();
        },
    });

    const exportStudyGroupCommentToCsv = async (): Promise<{ data: any }> => {
        studyGroupComments.refetch();

        if (!studyGroupComments.data) return { data: [] };

        const csvDataTemplate = studyGroupComments.data.map((comment: any) => {
            const strippedComment = comment.comment.replace(/(<([^>]+)>)/gi, '');
            return {
                fullName: comment.member.fullName,
                preferredName: comment.member.preferredName,
                taskDate: dateTransformer(comment.studyGroupTask.taskDate),
                comment: strippedComment,
            };
        });

        return { data: csvDataTemplate };
    };

    return (
        <div>
            <div className="flex justify-end">
                {staff?.role.STUDY_GROUP_TASK_CREATE && studyGroupData?.status === 'DRAFT' && (
                    <ConfirmationModal
                        title={t('generate-task')}
                        message={t('are-you-sure-you-want-to-generate-task?')}
                        okText={t('common:generate')}
                        onOk={() => generateTaskMutation.mutate()}
                    >
                        <Button
                            icon={<PlusCircleOutlined />}
                            type="primary"
                            onClick={() => generateTaskMutation.mutate()}
                            loading={generateTaskMutation.isLoading}
                        >
                            {t('generate-task')}
                        </Button>
                    </ConfirmationModal>
                )}
            </div>
            {studyGroupData?.status !== StudyGroupStatus.DRAFT && (
                <div className="flex justify-end">
                    <ExportData getExportData={exportStudyGroupCommentToCsv} fileName={`${studyGroupData?.name} - comments`} />
                </div>
            )}
            <div className="mt-4">
                <div className="mt-2 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                    {studyGroupTaskData &&
                        studyGroupTaskData?.map((task: StudyGroupTask) => {
                            const totalPenalties = task.penalties.reduce((acc: number, penalty: Penalty) => {
                                return acc + penalty.amount;
                            }, 0);

                            return (
                                <div
                                    key={task.id}
                                    onClick={() => {
                                        setIsPreviewTaskModal(true);
                                        setStudyGroupTaskId(task.id);
                                    }}
                                >
                                    <Card className="cursor-pointer h-fit overflow-y-auto" hoverable>
                                        <p className="text-sm md:text-base">
                                            <div className="mb-4">{dateTransformer(task.taskDate)}</div>
                                            <Divider />
                                            <Tooltip
                                                className="flex items-center gap-2 w-fit text-xs md:text-sm text-[#919191]"
                                                title={t('penalties')}
                                            >
                                                <AiOutlineExclamationCircle className="text-xs md:text-sm" />
                                                MYR {currencyFormatter(totalPenalties)}
                                            </Tooltip>
                                        </p>
                                    </Card>
                                </div>
                            );
                        })}
                </div>
                {!studyGroupTaskData?.length && <Empty description={t('no-task-available')} />}
            </div>
            <PreviewTask open={isPreviewTaskModal} setOpen={setIsPreviewTaskModal} />
        </div>
    );
};

export default Task;
