import { useStudyGroupContext } from '@/providers/StudyGroupContext';
import { AxiosErrorResponse, StudyGroupMembers } from '@/types';
import { conditionalReturn, errorMessageFormatter, stringAvatar, stringToColor } from '@/utils';
import { PlusCircleOutlined, WhatsAppOutlined } from '@ant-design/icons';
import { Avatar, Button, Card, Form, MenuProps, Pagination, Space } from 'antd';
import Link from 'next/link';
import AddMember from '../modals/AddMember';
import { useState } from 'react';
import { useTranslation } from 'next-i18next';
import { useQuery } from '@tanstack/react-query';
import { useRouter } from 'next/router';
import { toast } from 'react-toastify';
import FilterDrawer from '../modals/MemberFilter';
import { getStudyGroupMembers } from '@/services/study-group';
import usePagination from '@/hooks/usePagination';
import { useStaffContext } from '@/providers/StaffContext';
import ActionDropdown from '@/components/ui/ActionDropdown';
import DeleteStudyGroupMemberAction from '../DeleteStudyGroupMemberAction';
import { dateTransformer } from '@/utils/timeTransformer';
import ExportData from '../ExportMemberData';

const MemberTab: React.FC = () => {
    const { t } = useTranslation(['study-group', 'common', 'messages']);
    const { studyGroup, studyGroupMember } = useStudyGroupContext();
    const [isAddMemberModalOpen, setIsAddMemberModalOpen] = useState<boolean>(false);
    const router = useRouter();
    const { studyGroupId } = router.query;
    const [filterStudyGroupMemberForm] = Form.useForm();
    const [pagination, setPagination] = usePagination();
    const { staff } = useStaffContext();

    const studyGroupData = studyGroup.data;

    // Query
    const {
        data: studyGroupMemberData,
        refetch: studyGroupMemberRefetch,
        isLoading,
    } = useQuery({
        queryKey: ['study-group', studyGroupId, 'study-group-member', pagination],
        enabled: !!studyGroupId,
        queryFn: async () => {
            let searchedValue = filterStudyGroupMemberForm.getFieldsValue();

            const query = {
                ...searchedValue,
                page: pagination.page,
                pageSize: pagination.pageSize,
                sortField: pagination.sortField,
                sortOrder: pagination.sortOrder,
            };

            const res = await getStudyGroupMembers(studyGroupId as string, query);

            setPagination((prev) => {
                return {
                    ...prev,
                    page: res.data?.page,
                    total: res.data?.total,
                };
            });

            return res.data.rows;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Functions
    const exportStudyGroupMemberToCsv = async (): Promise<{ data: any }> => {
        studyGroupMember.refetch();

        if (!studyGroupMember.data) return { data: [] };

        const csvDataTemplate = studyGroupMember.data.studyGroupMembers.map((member: any) => {
            return {
                fullName: member.fullName,
                preferredName: member.preferredName,
                phoneNumber: member.phoneNumber,
                createdAt: dateTransformer(member.createdAt),
            };
        });

        return { data: csvDataTemplate };
    };

    // Function
    const onResetFilterHandler = () => {
        filterStudyGroupMemberForm.resetFields();
        studyGroupMemberRefetch();
    };

    return (
        <div className="w-full">
            <div className="flex justify-between gap-2 flex-col sm:flex-row">
                <div className="flex justify-between config_container mb-4">
                    <div className="flex flex-row">
                        <FilterDrawer
                            filterStudyGroupMemberForm={filterStudyGroupMemberForm}
                            onSearch={studyGroupMemberRefetch}
                            onReset={onResetFilterHandler}
                            loading={isLoading}
                        />
                        <Button type="link" className="list-none" onClick={onResetFilterHandler}>
                            {t('common:reset-filter')}
                        </Button>
                    </div>
                </div>
                <div className="flex flex-col sm:flex-row gap-2">
                    {studyGroupData?.status !== 'COMPLETED' && (
                        <Button icon={<PlusCircleOutlined />} type="primary" onClick={() => setIsAddMemberModalOpen(true)}>
                            {t('add-member')}
                        </Button>
                    )}
                    <ExportData query={filterStudyGroupMemberForm.getFieldsValue()} fileName={`${studyGroupData?.name} - Members`} />
                </div>
            </div>
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-2 mt-4">
                {studyGroupMemberData?.map((member: StudyGroupMembers) => {
                    const items: MenuProps['items'] = [
                        {
                            label: <Link href={`/member/${member.id}`}>{t('common:view')}</Link>,
                            key: 'view',
                            className: 'text-center',
                        },
                        ...conditionalReturn(!!staff?.role.BOOK_DELETE, [
                            {
                                label: <DeleteStudyGroupMemberAction memberId={member.id} />,
                                key: 'delete',
                                danger: true,
                                className: 'text-center',
                            },
                        ]),
                    ];

                    return (
                        <Card hoverable key={member.id}>
                            <div className="w-full font-semibold flex flex-row items-center gap-2 justify-between">
                                <div className="flex flex-row items-center gap-2">
                                    <div>
                                        <Avatar
                                            style={{ backgroundColor: stringToColor(member?.fullName) }}
                                            size={30}
                                            className="flex items-center flex-row gap-2"
                                        >
                                            {stringAvatar(member?.fullName)}
                                        </Avatar>
                                    </div>
                                    <Link href={`/member/${member.id}`}>
                                        <div className="text-xs sm:text-sm">
                                            {member?.fullName} {member?.preferredName && `(${member?.preferredName})`}
                                        </div>
                                    </Link>
                                </div>
                                <div className="flex flex-col min-[450px]:flex-row gap-2 sm:gap-6">
                                    <div className="flex items-center justify-center">
                                        <Link target="_blank" href={`https://wa.me/${member?.phoneNumber}`} onClick={(e) => e.stopPropagation()}>
                                            <WhatsAppOutlined className="text-lg text-green-500 hover:text-green-400 " />
                                        </Link>
                                    </div>
                                    <Space>
                                        <ActionDropdown items={items} />
                                    </Space>
                                </div>
                            </div>
                        </Card>
                    );
                })}
            </div>
            <div className="flex justify-end mt-4">
                <Pagination
                    current={pagination.page}
                    pageSize={pagination.pageSize}
                    defaultPageSize={1}
                    showSizeChanger={true}
                    pageSizeOptions={[10, 25, 50, 100]}
                    total={pagination.total}
                    showTotal={(total, range) => {
                        return t('common:pagination', { range0: range[0], range1: range[1], total: total });
                    }}
                    onChange={(page, pageSize) => setPagination({ ...pagination, page, pageSize })}
                />
            </div>
            <AddMember open={isAddMemberModalOpen} setOpen={setIsAddMemberModalOpen} refetch={() => studyGroupMember.refetch()} />
        </div>
    );
};

export default MemberTab;
