import { Button, Form, Modal } from 'antd';
import { useMutation } from '@tanstack/react-query';
import { useTranslation } from 'next-i18next';
import { AxiosErrorResponse, CreateStudyGroup } from '@/types';
import { toast } from 'react-toastify';
import { Dispatch, SetStateAction } from 'react';
import { errorMessageFormatter } from '@/utils';
import AddGroupForm from '../AddGroupForm';
import { addGroup } from '@/services/study-group';

interface AddGroupModalProps {
    open: boolean;
    setOpen: Dispatch<SetStateAction<boolean>>;
    refetch: () => void;
}

const AddGroup: React.FC<AddGroupModalProps> = ({ open, setOpen, refetch }) => {
    const { t } = useTranslation(['study-group', 'layout', 'common', 'messages']);
    const [addGroupForm] = Form.useForm();

    const addGroupMutation = useMutation({
        mutationFn: async (body: CreateStudyGroup) => {
            const response = await addGroup(body);
            return response;
        },
        onSuccess: () => {
            onCancelHandler();
            refetch();
        },
    });

    const onCancelHandler = () => {
        addGroupForm.resetFields();
        setOpen(false);
    };

    const onAddGroupHandler = async () => {
        addGroupForm.validateFields().then(async (values) => {
            toast.promise(addGroupMutation.mutateAsync(values), {
                pending: t('messages:adding-group'),
                success: t('messages:group-added'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    return (
        <Modal open={open} onCancel={onCancelHandler} footer={null} title={t('add-group')}>
            <Form form={addGroupForm} layout="vertical">
                <AddGroupForm />
                <div className="flex justify-between gap-4">
                    <Button loading={addGroupMutation.isLoading} block size="large" onClick={onCancelHandler}>
                        {t('common:cancel')}
                    </Button>
                    <Button loading={addGroupMutation.isLoading} block size="large" type="primary" onClick={onAddGroupHandler}>
                        {t('add-group')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default AddGroup;
