import { addPackage, updatePackage } from '@/services/site-setting';
import { AxiosErrorResponse, BaseModalProps } from '@/types';
import { Package, PackageParams } from '@/types/package';
import { errorMessageFormatter } from '@/utils';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { Editor } from '@tinymce/tinymce-react';
import { Button, Col, Form, Input, InputNumber, Modal, Row, Select } from 'antd';
import { useTranslation } from 'next-i18next';
import { useEffect } from 'react';
import { toast } from 'react-toastify';
import { v4 as uuid } from 'uuid';

interface PackageModalProps extends BaseModalProps {
    selectedPackage?: Package;
    setSelectedPackage: React.Dispatch<React.SetStateAction<Package | undefined>>;
}

const PackageModal: React.FC<PackageModalProps> = ({ selectedPackage, setSelectedPackage, open, setOpen }) => {
    const { t, i18n } = useTranslation(['site-setting', 'common', 'auth']);
    const currentLocale = i18n.language;
    const [packageForm] = Form.useForm<PackageParams>();
    const queryClient = useQueryClient();

    useEffect(() => {
        if (selectedPackage && open) {
            packageForm.setFieldsValue(selectedPackage);
        }
    }, [selectedPackage, open]);

    // Mutation
    const addPackageMutation = useMutation({
        mutationFn: async (params: PackageParams) => {
            const res = await addPackage(params);
            return res.data;
        },
        onSuccess: () => {
            setOpen(false);
            packageForm.resetFields();
            queryClient.invalidateQueries(['site-settings', 'packages', 'pagination']);
        },
    });

    const updatePackageMutation = useMutation({
        mutationFn: async (params: PackageParams) => {
            const res = await updatePackage(selectedPackage!.id, params);
            return res.data;
        },
        onSuccess: () => {
            setOpen(false);
            packageForm.resetFields();
            queryClient.invalidateQueries(['site-settings', 'packages', 'pagination']);
        },
    });

    const onAddPackageHandler = () => {
        packageForm.validateFields().then(async (values) => {
            toast.promise(addPackageMutation.mutateAsync(values), {
                pending: t('messages:adding-package'),
                success: t('messages:package-added'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    const onUpdatePackageHandler = () => {
        packageForm.validateFields().then(async (values) => {
            toast.promise(updatePackageMutation.mutateAsync(values), {
                pending: t('messages:updating-package'),
                success: t('messages:package-updated'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    const onCancelHandler = () => {
        setSelectedPackage(undefined);
        packageForm.resetFields();
        setOpen(false);
    };

    const breakPoint = {
        xs: 24,
        sm: 24,
        md: 12,
        lg: 12,
    };

    return (
        <Modal title={selectedPackage ? t('edit-package') : t('add-package')} open={open} onCancel={onCancelHandler} footer={null} width={650}>
            <Form
                form={packageForm}
                layout="vertical"
                size="large"
                name={selectedPackage ? 'edit_package_form' : 'add_package_form'}
                className="mt-2"
                initialValues={selectedPackage ? undefined : { status: '' }}
            >
                {/* Name Input */}
                <Form.Item
                    label={t('name')}
                    name="name"
                    rules={[
                        {
                            required: true,
                        },
                    ]}
                    className="mb-5"
                    labelCol={{ flex: '35px' }}
                >
                    <Input />
                </Form.Item>
                {/* Price Input */}
                <Form.Item
                    label={t('price')}
                    name="price"
                    rules={[
                        {
                            required: true,
                        },
                    ]}
                    labelCol={{ flex: '35px' }}
                >
                    <InputNumber className="w-full" precision={2} addonBefore={'MYR'} />
                </Form.Item>
                {/* Status Select */}
                <Row gutter={[16, 0]}>
                    <Col {...breakPoint}>
                        <Form.Item
                            name="status"
                            label={t('status')}
                            labelCol={{ flex: '35px' }}
                            rules={[
                                {
                                    required: true,
                                },
                            ]}
                        >
                            <Select placeholder={t('common:select')}>
                                <Select.Option value="">{t('select-a-status')}</Select.Option>
                                <Select.Option value="ACTIVE">{t('common:active')}</Select.Option>
                                <Select.Option value="INACTIVE">{t('common:inactive')}</Select.Option>
                            </Select>
                        </Form.Item>
                    </Col>
                    <Col {...breakPoint}>
                        <Form.Item
                            name="numberOfBookToken"
                            label={t('number-of-book-token')}
                            labelCol={{ flex: '35px' }}
                            rules={[
                                {
                                    required: true,
                                },
                            ]}
                        >
                            <InputNumber className="w-full" />
                        </Form.Item>
                    </Col>
                </Row>
                {/* Description */}
                <Form.Item name="description" label={t('description')} labelCol={{ flex: '35px' }} trigger="onEditorChange">
                    <Editor
                        id={uuid()}
                        tinymceScriptSrc={'/tinymce/js/tinymce/tinymce.min.js'}
                        init={{
                            height: 300,
                            menubar: '',
                            plugins: ['advlist', 'autolink', 'lists', 'link', 'searchreplace', 'code', 'preview', 'wordcount', 'table'],
                            toolbar:
                                'undo redo | blocks | ' +
                                'bold italic forecolor | link | alignleft aligncenter ' +
                                'alignright alignjustify | bullist numlist outdent indent | ' +
                                'fullscreen',
                            block_formats: 'Paragraph=p;Header 2=h2;Header 3=h3;Header 4=h4',
                            content_style: 'body { font-family:var(--font-ubuntu); font-size:14px; text-align:left }',
                            language: currentLocale === 'en-GB' ? undefined : currentLocale,
                            language_url: currentLocale === 'en-GB' ? undefined : '/editor-translation/' + currentLocale + '.js',
                            promotion: false,
                            paste_data_images: true,
                            contextmenu: false,
                        }}
                    />
                </Form.Item>
                <div className="flex justify-between mt-3 gap-x-3">
                    <Button type="default" block onClick={onCancelHandler}>
                        {t('common:cancel')}
                    </Button>
                    <Button
                        disabled={addPackageMutation.isLoading}
                        type="primary"
                        block
                        htmlType="submit"
                        onClick={selectedPackage ? onUpdatePackageHandler : onAddPackageHandler}
                    >
                        {selectedPackage ? t('common:edit') : t('common:add')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default PackageModal;
