import { ReactNode, useEffect, useState } from 'react';
import { useTranslation } from 'next-i18next';
import { Checkbox, Collapse, Form, FormInstance, Input } from 'antd';
import type { CheckboxChangeEvent } from 'antd/es/checkbox';
import RoleContainer from '@/components/role/FormContainer';
import roles from '@/data/role';

interface RoleFormProps {
    form: FormInstance;
    extraFormItem?: ReactNode;
    indeterminate?: boolean;
    checkAll?: boolean;
}

export const RoleForm: React.FC<RoleFormProps> = ({
    form,
    extraFormItem,
    indeterminate: indeterminateForUpdate = false,
    checkAll: checkAllForUpdate = false,
}) => {
    const { t } = useTranslation(['role']);

    // * Enable All Permissions
    const [indeterminate, setIndeterminate] = useState<boolean>(indeterminateForUpdate);
    const [checkAll, setCheckAll] = useState<boolean>(checkAllForUpdate);

    useEffect(() => {
        setCheckAll(checkAllForUpdate);
    }, [checkAllForUpdate]);

    useEffect(() => {
        setIndeterminate(indeterminateForUpdate);
    }, [indeterminateForUpdate]);

    const onFieldsChange = (changeFields: any, allFields: any) => {
        const fieldChanged = changeFields[0];

        // If "name" field is changed
        if (fieldChanged.name[0] === 'name') {
            return;
        }

        // Return only true or false without "checkAll", "reason", and "name" field
        const roles = allFields
            .filter((item: any) => !['checkAll', 'name', 'reason'].includes(item.name[0]))
            .map((item: any) => {
                return item.value;
            });

        // Calculate the number of true and false values in roles
        const trueCount = roles.filter((item: any) => item === true).length;
        const falseCount = roles.filter((item: any) => item === false).length;

        // Set "checkAll" field value based on the counts
        if (trueCount > 0 && falseCount > 0) {
            setCheckAll(false);
            setIndeterminate(true);
        } else {
            if (trueCount === 0) {
                setCheckAll(false);
                setIndeterminate(false);
            } else {
                setCheckAll(true);
                setIndeterminate(false);
            }
        }
    };

    const onCheckAllChange = (e: CheckboxChangeEvent) => {
        const checked = e.target.checked;
        let roleChecked: any = {};

        roles.map((role: string) => {
            roleChecked[role] = checked;
        });

        setIndeterminate(false);
        setCheckAll(checked);

        form.setFieldsValue(roleChecked);
    };

    const translatedView = t('common:view');
    const translatedCreate = t('common:create');
    const translatedUpdate = t('common:update');
    const translatedDelete = t('common:delete');

    return (
        <Form form={form} layout="vertical" name="role_form" onFieldsChange={onFieldsChange}>
            <Form.Item name="name" label={t('role-name')} rules={[{ required: true }]}>
                <Input />
            </Form.Item>
            <div className="mb-3">
                <Checkbox indeterminate={indeterminate} onChange={onCheckAllChange} checked={checkAll}>
                    {t('enable-all-permissions')}
                </Checkbox>
            </div>

            <Collapse
                defaultActiveKey={[
                    'staff',
                    'member',
                    'role',
                    'book',
                    'book_token',
                    'birthday',
                    'study_request',
                    'subscription',
                    'quotation',
                    'study_group',
                    'study_group_task',
                    'book_order',
                    'other',
                ]}
                className="mb-5"
            >
                {/* Staff */}
                <Collapse.Panel key="staff" header={<h3>{t('staff')}</h3>}>
                    <RoleContainer
                        roleItem={[
                            {
                                actionName: translatedView,
                                actionFormName: 'STAFF_VIEW',
                            },
                            {
                                actionName: translatedCreate,
                                actionFormName: 'STAFF_CREATE',
                            },
                            {
                                actionName: translatedUpdate,
                                actionFormName: 'STAFF_UPDATE',
                            },
                            {
                                actionName: translatedDelete,
                                actionFormName: 'STAFF_DELETE',
                            },
                        ]}
                    />
                </Collapse.Panel>
                {/* Agent */}
                <Collapse.Panel key="member" header={<h3>{t('member')}</h3>}>
                    <RoleContainer
                        roleItem={[
                            {
                                actionName: translatedView,
                                actionFormName: 'MEMBER_VIEW',
                            },
                            {
                                actionName: translatedCreate,
                                actionFormName: 'MEMBER_CREATE',
                            },
                            {
                                actionName: translatedUpdate,
                                actionFormName: 'MEMBER_UPDATE',
                            },
                            {
                                actionName: translatedDelete,
                                actionFormName: 'MEMBER_DELETE',
                            },
                        ]}
                    />
                </Collapse.Panel>
                {/* Role */}
                <Collapse.Panel key="role" header={<h3>{t('role')}</h3>}>
                    <RoleContainer
                        roleItem={[
                            {
                                actionName: translatedView,
                                actionFormName: 'ROLE_VIEW',
                            },
                            {
                                actionName: translatedCreate,
                                actionFormName: 'ROLE_CREATE',
                            },
                            {
                                actionName: translatedUpdate,
                                actionFormName: 'ROLE_UPDATE',
                            },
                            {
                                actionName: translatedDelete,
                                actionFormName: 'ROLE_DELETE',
                            },
                        ]}
                    />
                </Collapse.Panel>
                {/* Book */}
                <Collapse.Panel key="book" header={<h3>{t('book')}</h3>}>
                    <RoleContainer
                        roleItem={[
                            {
                                actionName: translatedView,
                                actionFormName: 'BOOK_VIEW',
                            },
                            {
                                actionName: translatedCreate,
                                actionFormName: 'BOOK_CREATE',
                            },
                            {
                                actionName: translatedUpdate,
                                actionFormName: 'BOOK_UPDATE',
                            },
                            {
                                actionName: translatedDelete,
                                actionFormName: 'BOOK_DELETE',
                            },
                        ]}
                    />
                </Collapse.Panel>
                {/* Book Token */}
                <Collapse.Panel key="book_token" header={<h3>{t('book-token')}</h3>}>
                    <RoleContainer
                        roleItem={[
                            {
                                actionName: translatedView,
                                actionFormName: 'BOOK_TOKEN_VIEW',
                            },
                            {
                                actionName: translatedCreate,
                                actionFormName: 'BOOK_TOKEN_CREATE',
                            },
                            {
                                actionName: translatedUpdate,
                                actionFormName: 'BOOK_TOKEN_UPDATE',
                            },
                            {
                                actionName: translatedDelete,
                                actionFormName: 'BOOK_TOKEN_DELETE',
                            },
                        ]}
                    />
                </Collapse.Panel>
                {/* Birthday */}
                <Collapse.Panel key="birthday" header={<h3>{t('birthday-calendar')}</h3>}>
                    <RoleContainer
                        roleItem={[
                            {
                                actionName: translatedView,
                                actionFormName: 'BIRTHDAY_VIEW',
                            },
                            {
                                actionName: translatedCreate,
                                actionFormName: 'BIRTHDAY_CREATE',
                            },
                            {
                                actionName: translatedUpdate,
                                actionFormName: 'BIRTHDAY_UPDATE',
                            },
                            {
                                actionName: translatedDelete,
                                actionFormName: 'BIRTHDAY_DELETE',
                            },
                        ]}
                    />
                </Collapse.Panel>
                {/* Study Request */}
                <Collapse.Panel key="study_request" header={<h3>{t('study-request')}</h3>}>
                    <RoleContainer
                        roleItem={[
                            {
                                actionName: translatedView,
                                actionFormName: 'STUDY_REQUEST_VIEW',
                            },
                            {
                                actionName: translatedUpdate,
                                actionFormName: 'STUDY_REQUEST_UPDATE',
                            },
                        ]}
                    />
                </Collapse.Panel>
                {/* Study Group */}
                <Collapse.Panel key="study_group" header={<h3>{t('study-group')}</h3>}>
                    <RoleContainer
                        roleItem={[
                            {
                                actionName: translatedView,
                                actionFormName: 'STUDY_GROUP_VIEW',
                            },
                            {
                                actionName: translatedCreate,
                                actionFormName: 'STUDY_GROUP_CREATE',
                            },
                            {
                                actionName: translatedUpdate,
                                actionFormName: 'STUDY_GROUP_UPDATE',
                            },
                            {
                                actionName: translatedDelete,
                                actionFormName: 'STUDY_GROUP_DELETE',
                            },
                        ]}
                    />
                </Collapse.Panel>
                {/* Study Group Task */}
                <Collapse.Panel key="study_group_task" header={<h3>{t('study-group-task')}</h3>}>
                    <RoleContainer
                        roleItem={[
                            {
                                actionName: translatedView,
                                actionFormName: 'STUDY_GROUP_TASK_VIEW',
                            },
                            {
                                actionName: translatedCreate,
                                actionFormName: 'STUDY_GROUP_TASK_CREATE',
                            },
                            {
                                actionName: translatedUpdate,
                                actionFormName: 'STUDY_GROUP_TASK_UPDATE',
                            },
                            {
                                actionName: translatedDelete,
                                actionFormName: 'STUDY_GROUP_TASK_DELETE',
                            },
                        ]}
                    />
                </Collapse.Panel>
                {/* Book Order */}
                <Collapse.Panel key="book_order" header={<h3>{t('book-order')}</h3>}>
                    <RoleContainer
                        roleItem={[
                            {
                                actionName: translatedView,
                                actionFormName: 'BOOK_ORDER_VIEW',
                            },
                            {
                                actionName: translatedCreate,
                                actionFormName: 'BOOK_ORDER_CREATE',
                            },
                            {
                                actionName: translatedUpdate,
                                actionFormName: 'BOOK_ORDER_UPDATE',
                            },
                            {
                                actionName: translatedDelete,
                                actionFormName: 'BOOK_ORDER_DELETE',
                            },
                        ]}
                    />
                </Collapse.Panel>
                {/* Quotation */}
                <Collapse.Panel key="quotation" header={<h3>{t('quotation')}</h3>}>
                    <RoleContainer
                        roleItem={[
                            {
                                actionName: translatedView,
                                actionFormName: 'QUOTATION_VIEW',
                            },
                            {
                                actionName: translatedCreate,
                                actionFormName: 'QUOTATION_CREATE',
                            },
                            {
                                actionName: translatedDelete,
                                actionFormName: 'QUOTATION_DELETE',
                            },
                        ]}
                    />
                </Collapse.Panel>
                {/* Other */}
                <Collapse.Panel key="other" header={<h3>{t('other')}</h3>}>
                    <RoleContainer
                        roleItem={[
                            {
                                actionName: t('site-setting'),
                                actionFormName: 'SITE_SETTING',
                            },
                            {
                                actionName: t('activity-log'),
                                actionFormName: 'ACTIVITY_LOG',
                            },
                        ]}
                    />
                </Collapse.Panel>
            </Collapse>

            {extraFormItem}
        </Form>
    );
};
