import { useEffect } from 'react';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { UseQueryResult, useMutation } from '@tanstack/react-query';
import { Button, Form, Input, Modal } from 'antd';
import { updateMyEmail } from '@/services/my-profile';
import { AxiosErrorResponse, BaseModalProps, Staff, UpdateStaffEmailParams } from '@/types';
import { errorMessageFormatter } from '@/utils/errorFormatter';

interface EmailModalProps extends BaseModalProps {
    staff: Staff;
    staffQuery: UseQueryResult<Staff | null>;
}

const EmailModal: React.FC<EmailModalProps> = ({ open, setOpen, staff, staffQuery }) => {
    const { t } = useTranslation(['my-profile', 'layout', 'common', 'messages']);
    const [form] = Form.useForm<{ email: string }>();
    const emailInput = Form.useWatch('email', form);

    useEffect(() => {
        if (open) {
            form.setFieldsValue({ email: staff.email });
        }
    }, [open]);

    const isDiasbled = () => {
        if (!emailInput || emailInput === staff.email) {
            return true;
        }

        return false;
    };

    // Mutations
    const updateMyEmailMutation = useMutation({
        mutationFn: async (body: UpdateStaffEmailParams) => {
            const res = await updateMyEmail(body.email);
            return res.data;
        },
        onSuccess: () => {
            setOpen(false);
            staffQuery.refetch();
        },
    });

    // Functions
    const onCancelHandler = () => {
        form.resetFields();
        setOpen(false);
    };

    const onUpdateStaffEmailHandler = () => {
        form.validateFields().then(async (values) => {
            toast.promise(updateMyEmailMutation.mutateAsync(values), {
                pending: t('messages:updating-staff-email'),
                success: t('messages:staff-email-updated'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    return (
        <Modal open={open} onCancel={onCancelHandler} footer={null} title={t('edit-email')}>
            <Form layout="vertical" form={form} className="mt-4">
                <Form.Item
                    name="email"
                    rules={[
                        { required: true },
                        {
                            type: 'email',
                        },
                    ]}
                >
                    <Input size="large" />
                </Form.Item>
                <div className="pt-2 flex gap-3">
                    <Button loading={updateMyEmailMutation.isLoading} size="large" block onClick={onCancelHandler}>
                        {t('common:cancel')}
                    </Button>
                    <Button
                        loading={updateMyEmailMutation.isLoading}
                        size="large"
                        type="primary"
                        disabled={isDiasbled()}
                        block
                        onClick={onUpdateStaffEmailHandler}
                    >
                        {t('common:update')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default EmailModal;
