import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { useQuery } from '@tanstack/react-query';
import { Button, Form, Spin, Table, TableColumnProps, Tag } from 'antd';
import { useStaffContext } from '@/providers/StaffContext';
import { AxiosErrorResponse, StudyGroup } from '@/types';
import { errorMessageFormatter } from '@/utils/errorFormatter';
import router from 'next/router';
import FilterDrawer from '@/components/study-group/modals/filter';
import usePagination from '@/hooks/usePagination';
import Link from 'next/link';
import { dateTimeTransformer, dateTransformer } from '@/utils/timeTransformer';
import { getStudyGroupList } from '@/services/study-group';

const MemberStudyGroup: React.FC = () => {
    const { t } = useTranslation(['member', 'common']);
    const { staff } = useStaffContext();
    const [pagination, setPagination, paginationOnChange] = usePagination();
    const [filterStudyGroupForm] = Form.useForm();

    const { memberId } = router.query;

    // Query
    const { data, isFetching, refetch } = useQuery({
        queryKey: ['member', 'study-group', 'pagination', pagination],
        queryFn: async () => {
            const filterValues = filterStudyGroupForm.getFieldsValue();

            const query = {
                ...filterValues,
                memberId,
                page: pagination.page,
                pageSize: pagination.pageSize,
                sortField: pagination.sortField,
                sortOrder: pagination.sortOrder,
            };

            const response = await getStudyGroupList(query);

            setPagination((prev) => {
                return {
                    ...prev,
                    page: response.data?.page,
                    total: response.data?.total,
                };
            });

            return response.data?.rows;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Functions
    const onResetFilterHandler = () => {
        filterStudyGroupForm.resetFields();
        refetch();
    };

    // Data
    const columns: TableColumnProps<StudyGroup>[] = [
        {
            title: t('group-name'),
            dataIndex: 'name',
            key: 'name',
            render: (_: unknown, studyGroup) => {
                return (
                    <Link href={`/study-group/${studyGroup.id}`}>
                        <b>{studyGroup.name}</b>
                    </Link>
                );
            },
        },
        {
            title: t('book-name'),
            dataIndex: 'book',
            key: 'book',
            render: (_: unknown, studyGroup) => {
                return <Link href={`/book/${studyGroup.book.id}`}>{studyGroup.book?.name}</Link>;
            },
        },
        {
            title: t('start-date'),
            dataIndex: 'startDate',
            key: 'startDate',
            render: (_: unknown, studyGroup) => {
                // Format to malaysia timezone
                return dateTransformer(studyGroup.startDate);
            },
        },
        {
            title: t('end-date'),
            dataIndex: 'endDate',
            key: 'endDate',
            render: (_: unknown, studyGroup) => {
                return dateTransformer(studyGroup.endDate);
            },
        },
        {
            title: t('status'),
            dataIndex: 'status',
            key: 'status',
            width: 130,
            render: (_: unknown, studyGroup) => {
                let tagColor, tagText;

                switch (studyGroup?.status) {
                    case 'DRAFT':
                        tagColor = '#CCCCCC';
                        tagText = t('draft');
                        break;
                    case 'GENERATED':
                        tagColor = '#0074D9';
                        tagText = t('generated');
                        break;
                    case 'ONGOING':
                        tagColor = '#FFD700';
                        tagText = t('ongoing');
                        break;
                    case 'COMPLETED':
                        tagColor = '#2ECC40';
                        tagText = t('completed');
                        break;
                }

                return <Tag color={tagColor}>{tagText}</Tag>;
            },
        },
        {
            title: t('common:created-at'),
            dataIndex: 'createdAt',
            key: 'createdAt',
            width: 200,
            render: (_: unknown, studyGroup) => {
                return dateTimeTransformer(studyGroup.createdAt);
            },
        },
    ];

    return (
        <Spin spinning={isFetching}>
            {staff && (
                <div className="flex flex-col lg:flex-row">
                    <div className="order-2 w-full px-2 mt-2 lg:mt-z0 lg:flex-1 lg:order-1">
                        <div className="flex justify-between mb-4 flex-col sm:flex-row">
                            <div className="flex">
                                <FilterDrawer
                                    filterStudyGroupForm={filterStudyGroupForm}
                                    onSearch={refetch}
                                    onReset={onResetFilterHandler}
                                    loading={isFetching}
                                />
                                <Button type="link" className="list-none" onClick={onResetFilterHandler}>
                                    {t('common:reset-filter')}
                                </Button>
                            </div>
                        </div>
                        <div className="table_container">
                            <Table
                                columns={columns}
                                dataSource={data}
                                loading={isFetching}
                                rowKey={(record) => record.id}
                                scroll={{ x: 1000 }}
                                onChange={paginationOnChange}
                                pagination={{
                                    current: pagination.page,
                                    pageSize: pagination.pageSize,
                                    defaultPageSize: 1,
                                    showSizeChanger: true,
                                    pageSizeOptions: [10, 25, 50, 100],
                                    showTotal: (total, range) => {
                                        return t('common:pagination', { range0: range[0], range1: range[1], total: total });
                                    },
                                    total: pagination.total,
                                }}
                            />
                        </div>
                    </div>
                </div>
            )}
        </Spin>
    );
};

export default MemberStudyGroup;
