import { CloudDownloadOutlined } from '@ant-design/icons';
import { Button } from 'antd';
import { useTranslation } from 'next-i18next';
import { useQuery, useQueryClient } from '@tanstack/react-query';
import { AxiosErrorResponse } from '@/types';
import { errorMessageFormatter } from '@/utils';
import { toast } from 'react-toastify';
import { getExportMemberList } from '@/services/member';
import ExcelJS from 'exceljs';
import { dateTimeTransformer, dateTransformer } from '@/utils/timeTransformer';

const ExportData: React.FC = () => {
    const { t } = useTranslation(['member']);
    const queryClient = useQueryClient();

    // Query
    useQuery({
        queryKey: ['member', 'export'],
        enabled: false,
        queryFn: async () => {
            const res = await getExportMemberList();

            return res.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    const workbook = new ExcelJS.Workbook();
    const worksheet = workbook.addWorksheet('Members');

    // Columns
    worksheet.columns = [
        { header: 'Full Name', key: 'fullName', width: 20 },
        { header: 'Address', key: 'address', width: 40 },
        { header: 'Country', key: 'country', width: 20 },
        { header: 'State', key: 'state', width: 20 },
        { header: 'City', key: 'city', width: 20 },
        { header: 'Date of Birth', key: 'dateOfBirth', width: 20 },
        { header: 'Phone Number', key: 'phoneNumber', width: 20 },
        { header: 'Created At', key: 'createdAt', width: 20 },
    ];

    // Add Row Style
    worksheet.getRow(1).font = { bold: true };
    worksheet.getRow(1).alignment = { horizontal: 'center' };
    worksheet.getRow(1).fill = {
        type: 'pattern',
        pattern: 'solid',
        fgColor: { argb: 'FFA9A9A9' },
    };

    const generateExcelHandler = async (data: any) => {
        data?.map((member: any) => {
            worksheet.addRow({
                fullName: member.fullName,
                address: member.address,
                country: member.country,
                state: member.state,
                city: member.city,
                dateOfBirth: dateTransformer(member.dateOfBirth),
                phoneNumber: `;${member.phoneNumber}`,
                createdAt: dateTimeTransformer(member.createdAt),
            });
        });

        // Generate the Excel file
        await workbook.xlsx.writeBuffer().then((buffer: any) => {
            const blob = new Blob([buffer], { type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' });
            const link = document.createElement('a');
            const url = URL.createObjectURL(blob);
            link.href = url;
            link.setAttribute('download', `member.csv`);
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        });
    };

    const exportToCsvHandler = () => {
        queryClient.fetchQuery(['member', 'export']).then((res) => {
            generateExcelHandler(res);
        });
    };

    return (
        <Button type="primary" icon={<CloudDownloadOutlined />} onClick={exportToCsvHandler}>
            {t('export-csv')}
        </Button>
    );
};

export default ExportData;
