import axiosInstance, { uploadInstance } from '@/configs/service';
import { MediaFormValue, MediaResponse, PaginationResponse } from '@/types';
import {
    CommentLike,
    MedicalCertificate,
    Replies,
    StudyGroup,
    StudyGroupMember,
    StudyGroupMemberTaskComment,
    StudyGroupTask,
    StudyGroupTaskComment,
    UploadMedicalCertificatePayload,
} from '@/types/study-group';
import { mediaUrlGenerator } from '@/utils';

// Context Provider
export const getStudyGroupById = (studyGroupId: string) => {
    return axiosInstance.get<StudyGroup>(`/member/study-group/${studyGroupId}/id`);
};

export const getStudyGroupMember = (studyGroupId: string) => {
    return axiosInstance.get<StudyGroupMember>(`/member/study-group/${studyGroupId}/member`);
};

export const getStudyGroupTaskById = (studyGroupTaskId: string) => {
    return axiosInstance.get<StudyGroupTask>(`/member/study-group/${studyGroupTaskId}/task-by-id`);
};

///////////////////
// Study Group
///////////////////

// Get
export const getStudyGroupList = (query: StudyGroup) => {
    return axiosInstance.get<PaginationResponse<StudyGroup>>('/member/study-group', { params: query });
};

export const getMemberPenalty = (studyGroupId: string) => {
    return axiosInstance.get(`/member/study-group/${studyGroupId}/member-penalty`);
};

///////////////////
// Study Group Task
///////////////////

// Get
export const getMedicalCertificate = (studyGroupTaskId: string) => {
    return axiosInstance.get<MedicalCertificate>(`/member/study-group/${studyGroupTaskId}/medical-certificate`);
};

// Create
export const uploadMedicalCertificateMedia = async (url: string, option: any) => {
    const formData = new FormData();
    formData.append('file', option.file);

    try {
        const response = await uploadInstance.post<MediaResponse>(url, formData);

        option.onSuccess({
            uid: response.data.id,
            name: response.data.name,
            url: mediaUrlGenerator(response.data.key),
            status: 'done',
        } as MediaFormValue);
    } catch (error) {
        option.onError(error);
    }
};

export const uploadMedicalCertificate = async (studyGroupTaskId: string, data: UploadMedicalCertificatePayload) => {
    return axiosInstance.post(`/member/study-group/${studyGroupTaskId}`, data);
};

///////////////////
// Comment
///////////////////

// Get
export const getStudyGroupTaskComments = (studyGroupTaskId: string, query: any) => {
    return axiosInstance.get<PaginationResponse<StudyGroupTaskComment>>(`/member/study-group/${studyGroupTaskId}/task-comments`, {
        params: query,
    });
};

export const getMemberStudyGroupComments = (studyGroupId: string) => {
    return axiosInstance.get<StudyGroupTaskComment[]>(`/member/study-group/${studyGroupId}/member-comments`);
};

export const getOwnTaskComment = (studyGroupTaskId: string) => {
    return axiosInstance.get<StudyGroupTaskComment>(`/member/study-group/${studyGroupTaskId}/own-task-comment`);
};

export const getMemberTaskComments = (query: StudyGroup) => {
    return axiosInstance.get<StudyGroupMemberTaskComment[]>('/member/study-group/member-task-comments', { params: query });
};

export const getTaskCommentLikes = (studyGroupTaskId: string) => {
    return axiosInstance.get<CommentLike[]>(`/member/study-group/${studyGroupTaskId}/like`);
};

export const getReplies = (studyGroupTaskId: string, commentId: string) => {
    return axiosInstance.get<Replies[]>(`/member/study-group/${studyGroupTaskId}/reply/${commentId}`);
};

// Update
export const updateComment = (studyGroupTaskId: string, comments: string) => {
    return axiosInstance.put(`/member/study-group/${studyGroupTaskId}/comment`, { comment: comments });
};

export const updateReply = (studyGroupTaskId: string, reply: string, commentId?: string) => {
    return axiosInstance.put(`/member/study-group/${studyGroupTaskId}/reply`, { reply, commentId });
};

export const scheduleComment = (studyGroupTaskId: string, values: string, prePlannedDate: string) => {
    return axiosInstance.post(`/member/study-group/${studyGroupTaskId}/schedule-comment`, { comment: values, prePlannedDate });
};

export const likeComment = (studyGroupTaskCommentId: string, memberId: string, studyGroupTaskId: string) => {
    return axiosInstance.post(`/member/study-group/${studyGroupTaskId}/like/${studyGroupTaskCommentId}`, { memberId });
};
