import { FILE_SIZE_LIMIT } from '@/configs';
import { useStudyGroupContext } from '@/providers/StudyGroupContext';
import { uploadMedicalCertificate, uploadMedicalCertificateMedia } from '@/services/study-group';
import { mediaFormValueToResponse } from '@/transformer/media/media';
import { AxiosErrorResponse } from '@/types';
import { errorMessageFormatter, normFile } from '@/utils';
import { UploadOutlined } from '@ant-design/icons';
import { useMutation } from '@tanstack/react-query';
import { Button, Form, Modal, Upload } from 'antd';
import { useTranslation } from 'next-i18next';
import { Dispatch, SetStateAction } from 'react';
import { toast } from 'react-toastify';
import { UploadMedicalCertificatePayload } from '@/types/study-group';

interface MedicalLeaveProps {
    open: boolean;
    setOpen: Dispatch<SetStateAction<boolean>>;
    medicalCertificateRefetch: () => void;
}

const MedicalCertificate: React.FC<MedicalLeaveProps> = ({ open, setOpen, medicalCertificateRefetch }) => {
    const { t } = useTranslation(['study-group', 'layout', 'common', 'messages']);
    const [applyMedicalCertificateForm] = Form.useForm();
    const { studyGroupTask, studyGroup, studyGroupTaskId } = useStudyGroupContext();

    // Mutation
    const uploadMedicalCertificateMutation = useMutation({
        mutationFn: async (data: UploadMedicalCertificatePayload) => {
            const response = await uploadMedicalCertificate(studyGroupTaskId, data);

            return response;
        },
        onSuccess: () => {
            studyGroup.refetch();
            studyGroupTask.refetch();
            medicalCertificateRefetch();
            onCancelHandler();
        },
    });

    // Function
    const onCancelHandler = () => {
        applyMedicalCertificateForm.resetFields();
        setOpen(false);
    };

    const onUploadMedicalCertificateHandler = () => {
        applyMedicalCertificateForm.validateFields().then(async (values) => {
            const formattedData = values.uploadFile.map(mediaFormValueToResponse);
            const body: UploadMedicalCertificatePayload = {
                medias: formattedData,
            };

            toast.promise(uploadMedicalCertificateMutation.mutateAsync(body), {
                pending: t('messages:uploading-medical-certificate'),
                success: t('messages:medical-certificate-uploaded'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    return (
        <Modal title={t('apply-medical-certificate')} open={open} onCancel={onCancelHandler} footer={null}>
            <Form form={applyMedicalCertificateForm} layout="vertical">
                <Form.Item
                    name="uploadFile"
                    label={t('upload-file')}
                    rules={[{ required: true, message: t('messages:required') }]}
                    getValueFromEvent={normFile}
                    valuePropName="fileList"
                >
                    <Upload
                        className="!w-full"
                        maxCount={3}
                        beforeUpload={(file) => {
                            const isLTSizeLimit = file.size / 1024 / 1024 > FILE_SIZE_LIMIT;

                            if (isLTSizeLimit) {
                                toast.error(t('messages:file-too-large', { fileName: file.name, size: FILE_SIZE_LIMIT }));
                                return Upload.LIST_IGNORE;
                            }

                            return true;
                        }}
                        customRequest={(options) => uploadMedicalCertificateMedia('/member/study-group/upload', options)}
                    >
                        <Button block icon={<UploadOutlined />} size="large">
                            {t('common:upload-medical-certificate')}
                        </Button>
                    </Upload>
                </Form.Item>
            </Form>
            <div className="flex flex-col md:flex-row gap-4">
                <Button block onClick={onCancelHandler} loading={uploadMedicalCertificateMutation.isLoading}>
                    {t('common:cancel')}
                </Button>
                <Button block type="primary" onClick={onUploadMedicalCertificateHandler} loading={uploadMedicalCertificateMutation.isLoading}>
                    {t('common:submit')}
                </Button>
            </div>
        </Modal>
    );
};

export default MedicalCertificate;
