import { AxiosErrorResponse } from '@/types';
import { Card, Descriptions, Spin } from 'antd';
import PreviewBook from '../PreviewBook';
import { useTranslation } from 'next-i18next';
import { useRouter } from 'next/router';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { bookKeys, getBookById, updateBookRequestStatus } from '@/services/book';
import { errorMessageFormatter } from '@/utils';
import { toast } from 'react-toastify';
import Initialize from '@/components/globals/Initialize';
import React from 'react';
import { dateTimeTransformer } from '@/utils/timeTransformer';
const { Item } = Descriptions;

const BookDetailTab: React.FC = () => {
    const { t } = useTranslation(['book', 'layout', 'common', 'messages']);
    const router = useRouter();
    const { bookId } = router.query;
    const queryClient = useQueryClient();

    // Query
    const getBookDetailsQuery = useQuery({
        queryKey: bookKeys.id(bookId as string),
        queryFn: async () => {
            const response = await getBookById(bookId as string);
            return response.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    //Mutations
    const updateBookRequestStatusMutation = useMutation({
        mutationFn: async (bookId: string) => {
            const res = await updateBookRequestStatus(bookId);
            return res.data;
        },
        onSuccess: () => {
            queryClient.invalidateQueries(['member', 'book']);
        },
    });

    // Function
    const onBookRequestStatusHandler = async (bookId: string) => {
        toast.promise(updateBookRequestStatusMutation.mutateAsync(bookId), {
            pending: t('messages:updating-book-request-status'),
            success: t('messages:book-request-status-updated'),
            error: {
                render({ data }: any) {
                    return t(errorMessageFormatter(data));
                },
            },
        });
    };

    if (getBookDetailsQuery.isLoading) {
        return <Initialize />;
    }

    return (
        <Spin spinning={getBookDetailsQuery.isLoading}>
            {getBookDetailsQuery.data && (
                <div className="flex flex-col lg:flex-row gap-8">
                    <div className="flex-1">
                        <Card>
                            <PreviewBook book={getBookDetailsQuery.data} />
                        </Card>
                    </div>
                    <div className="lg:basis-1/4">
                        <Card>
                            <Descriptions layout="vertical" size="small" column={1}>
                                <Item contentStyle={{ marginBottom: '15px' }} label={t('common:created-at')}>
                                    {dateTimeTransformer(getBookDetailsQuery.data.createdAt)}
                                </Item>
                            </Descriptions>
                        </Card>
                    </div>
                </div>
            )}
        </Spin>
    );
};

export default BookDetailTab;
