import React, { useState } from 'react';
import { useTranslation } from 'next-i18next';
import { FilterOutlined, SearchOutlined } from '@ant-design/icons';
import { Button, Col, Form, Input, Modal, Row, Select } from 'antd';
import { FormInstance } from 'antd/es/form/Form';

interface FilterAttributes {
    filterBookOrderRequestForm: FormInstance;
    onReset: () => void;
    onSearch: () => void;
    loading: boolean;
}

const FilterDrawer: React.FC<FilterAttributes> = ({ filterBookOrderRequestForm, onReset, onSearch, loading }) => {
    const { t } = useTranslation(['book-token', 'staff', 'common']);
    const criteriaSelected = filterBookOrderRequestForm.getFieldsValue();
    const criteriaCount: number = Object.values(criteriaSelected).reduce((count: number, val) => count + (val ? 1 : 0), 0);
    const [isModalOpen, setIsModalOpen] = useState(false);

    const breakPoint = {
        xs: 24,
        sm: 24,
        md: 12,
        lg: 12,
    };

    const onResetHandler = () => {
        onReset();
        setIsModalOpen(false);
    };

    const onSearchHandler = () => {
        onSearch();
        setIsModalOpen(false);
    };

    return (
        <div>
            <Button onClick={() => setIsModalOpen(true)} icon={<FilterOutlined />}>
                {criteriaCount > 0 && `(${criteriaCount})`} {t('common:filter')}
            </Button>
            <Modal title={t('common:filter')} onCancel={onResetHandler} open={isModalOpen} footer={null} width={650}>
                <Form form={filterBookOrderRequestForm} layout="vertical" name="filter_form" className="mt-6" size="large">
                    <Row gutter={[16, 0]}>
                        {/* Ref No */}
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="refNo" label={t('ref-no')} labelCol={{ flex: '35px' }}>
                                <Input placeholder={t('enter-ref-no')} />
                            </Form.Item>
                        </Col>
                        {/* Member Name */}
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="memberName" label={t('member-name')} labelCol={{ flex: '35px' }}>
                                <Input placeholder={t('enter-member-name')} />
                            </Form.Item>
                        </Col>
                        {/* Book Name */}
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="bookName" label={t('book-name')} labelCol={{ flex: '35px' }}>
                                <Input placeholder={t('enter-book-name')} />
                            </Form.Item>
                        </Col>
                        {/* Status */}
                        <Col {...breakPoint}>
                            <Form.Item initialValue="" name="status" label={t('status')} labelCol={{ flex: '35px' }}>
                                <Select placeholder={t('common:select')}>
                                    <Select.Option value="">{t('select-a-status')}</Select.Option>
                                    <Select.Option value="PENDING">{t('pending')}</Select.Option>
                                    <Select.Option value="PAID">{t('paid')}</Select.Option>
                                    <Select.Option value="UNPAID">{t('unpaid')}</Select.Option>
                                </Select>
                            </Form.Item>
                        </Col>
                    </Row>
                    <div className="flex justify-between mt-3 gap-x-3">
                        <Button onClick={onResetHandler} block loading={loading}>
                            {t('common:reset')}
                        </Button>
                        <Button type="primary" onClick={onSearchHandler} block loading={loading}>
                            {t('common:apply-filter')}
                            <SearchOutlined />
                        </Button>
                    </div>
                </Form>
            </Modal>
        </div>
    );
};

export default FilterDrawer;
