import { Trans, useTranslation } from 'next-i18next';
import { AxiosErrorResponse } from '@/types';
import { toast } from 'react-toastify';
import { useMutation } from '@tanstack/react-query';
import { errorMessageFormatter } from '@/utils';
import { Button } from 'antd';
import { BookOrderList } from '@/types/book-order';
import ConfirmationModal from '@/components/modals/ConfirmationModal';
import { cancelBookOrderById } from '@/services/book-order';

interface CancelBookOrderModalProps {
    bookOrder: BookOrderList;
    refetch: () => void;
}

const CancelBookOrderModal: React.FC<CancelBookOrderModalProps> = ({ bookOrder, refetch }) => {
    const { t } = useTranslation(['book-order', 'layout', 'common', 'messages']);

    const deleteBookOrderMutation = useMutation({
        mutationFn: async () => {
            const response = await cancelBookOrderById(bookOrder.id);
            return response.data;
        },
        onSuccess: () => {
            refetch();
        },
    });

    const onDeleteHandler = () => {
        toast.promise(deleteBookOrderMutation.mutateAsync(), {
            pending: t('messages:cancelling-book-order'),
            success: t('messages:book-order-cancelled'),
            error: {
                render({ data }) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    return (
        <ConfirmationModal
            title={t('common:cancel-order-confirmation')}
            message={<Trans i18nKey={'messages:are-you-sure-you-want-to-cancel-this-book-order'} components={{ strong: <strong /> }} />}
            okText={t('common:cancel-order')}
            onOk={onDeleteHandler}
            reason={false}
            width={400}
        >
            <Button type="text" block loading={deleteBookOrderMutation.isLoading}>
                {t('common:cancel-order')}
            </Button>
        </ConfirmationModal>
    );
};

export default CancelBookOrderModal;
