import { orderBook } from '@/services/book-order';
import { AxiosErrorResponse, BaseModalProps, Book } from '@/types';
import { OrderBookParams } from '@/types/book-order';
import { errorMessageFormatter } from '@/utils';
import { currencyFormatter } from '@/utils/currencyFormatter';
import { useMutation } from '@tanstack/react-query';
import { Button, Form, Input, InputNumber, Modal } from 'antd';
import { useTranslation } from 'next-i18next';
import { useEffect } from 'react';
import { toast } from 'react-toastify';

interface AddBookOrderModalProps extends BaseModalProps {
    book: Book;
}

const AddBookOrderModal: React.FC<AddBookOrderModalProps> = ({ book, open, setOpen }) => {
    const { t } = useTranslation(['book-order', 'messages']);
    const [addBookOrderForm] = Form.useForm();
    const quantity = Form.useWatch('quantity', addBookOrderForm);
    const price = Form.useWatch('pricePerBook', addBookOrderForm);

    useEffect(() => {
        if (open && !!book) {
            addBookOrderForm.setFieldsValue({
                name: book.name,
                pricePerBook: book.price,
            });
        }
    }, [book]);

    useEffect(() => {
        addBookOrderForm.setFieldsValue({
            totalPrice: quantity * price || 0,
        });
    }, [quantity, price]);

    // Mutations
    const addBookOrderMutation = useMutation({
        mutationFn: async (data: OrderBookParams) => {
            const res = await orderBook(data);
            return res.data;
        },
        onSuccess() {
            onCancelHandler();
        },
    });

    const onCancelHandler = () => {
        addBookOrderForm.resetFields();
        setOpen(false);
    };

    const addBookOrderHandler = () => {
        addBookOrderForm.validateFields().then(async (values) => {
            const { quantity } = values;

            const payload = {
                quantity,
                bookId: book.id,
            };

            toast.promise(addBookOrderMutation.mutateAsync(payload), {
                pending: t('messages:ordering-book'),
                success: t('messages:book-ordered'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    return (
        <Modal open={open} title={t('order-book')} footer={null} onCancel={onCancelHandler}>
            <Form form={addBookOrderForm} layout="vertical">
                <Form.Item label={t('book')} name={'name'}>
                    <Input disabled />
                </Form.Item>
                <Form.Item label={t('quantity')} name={'quantity'} rules={[{ required: true }]}>
                    <InputNumber min={0} className="w-full" />
                </Form.Item>
                <Form.Item label={t('price-per-book')} name={'pricePerBook'}>
                    <InputNumber disabled formatter={currencyFormatter} className="w-full" addonBefore={'MYR'} />
                </Form.Item>
                <Form.Item label={t('total-price')} name={'totalPrice'}>
                    <InputNumber disabled formatter={currencyFormatter} className="w-full" addonBefore={'MYR'} />
                </Form.Item>
            </Form>
            <div className="flex flex-row gap-4">
                <Button block onClick={onCancelHandler} loading={addBookOrderMutation.isLoading}>
                    {t('common:cancel')}
                </Button>
                <Button block type="primary" onClick={addBookOrderHandler} loading={addBookOrderMutation.isLoading}>
                    {t('order-book')}
                </Button>
            </div>
        </Modal>
    );
};

export default AddBookOrderModal;
