import { Body, Controller, Get, HttpCode, HttpStatus, Post, Query, Put, Param, Delete, Req } from '@nestjs/common';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import {
    createStaffSchema,
    staffEmailSchema,
    staffQuerySchema,
    updateStaffEmailSchema,
    updateStaffPasswordSchema,
    updateStaffPhoneNumberSchema,
    updateStaffProfileSchema,
    updateStaffRoleSchema,
} from './staff.validator';
import { StaffService } from './staff.service';
import {
    CreateStaffDto,
    StaffEmailDto,
    UpdateStaffEmailDto,
    UpdateStaffPasswordDto,
    UpdateStaffPhoneNumberDto,
    UpdateStaffProfileDto,
    UpdateStaffRoleDto,
} from './staff.dto';
import { Authenticator, Authorizer, Staff } from 'src/auth/auth.decorator';
import { StaffInRequest, StaffQuery, User } from 'src/types';
import { ActivityLogService } from 'src/activity-log/activity-log.service';

@Controller()
export class StaffController {
    constructor(
        private staffService: StaffService,
        private activityLogService: ActivityLogService,
    ) {}

    @Authenticator(User.STAFF)
    @Authorizer('STAFF_VIEW')
    @Get('')
    @HttpCode(HttpStatus.OK)
    async getStaff(@Query(new JoiValidationPipe(staffQuerySchema)) query: StaffQuery) {
        const staffList = await this.staffService.getStaff(query);

        return {
            total: staffList.count,
            rows: staffList.rows,
            page: staffList.page,
        };
    }

    @Authenticator(User.STAFF)
    @Authorizer('STAFF_VIEW')
    @Get(':staffId')
    @HttpCode(HttpStatus.OK)
    async getStaffById(@Param('staffId') staffId: string) {
        const response = await this.staffService.getStaffById(staffId);
        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STAFF_CREATE')
    @Post('')
    @HttpCode(HttpStatus.OK)
    async createStaff(@Staff() staff: StaffInRequest, @Body(new JoiValidationPipe(createStaffSchema)) body: CreateStaffDto) {
        const { id: staffId, fullName } = staff;
        const response = await this.staffService.createStaff(body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'staff',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'CREATE',
            description: 'api-messages:create-staff',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STAFF_UPDATE')
    @Put(':staffId/status')
    @HttpCode(HttpStatus.OK)
    async updateStaffStatus(@Staff() staff: StaffInRequest, @Param('staffId') staffId: string) {
        const { id: myStaffId, fullName } = staff;
        const response = await this.staffService.updateStaffStatus(staffId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'staff',
                id: response.id,
            },
            staffId: myStaffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-staff-status',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STAFF_UPDATE')
    @Put(':staffId/role')
    @HttpCode(HttpStatus.OK)
    async updateStaffRole(
        @Staff() staff: StaffInRequest,
        @Param('staffId') staffId: string,
        @Body(new JoiValidationPipe(updateStaffRoleSchema)) body: UpdateStaffRoleDto,
    ) {
        const { id: myStaffId, fullName } = staff;
        const { roleId } = body;

        const response = await this.staffService.updateStaffRole(myStaffId, staffId, roleId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'staff',
                id: response.id,
            },
            staffId: myStaffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-staff-role',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STAFF_UPDATE')
    @Put(':staffId/profile')
    @HttpCode(HttpStatus.OK)
    async updateStaffProfile(
        @Staff() staff: StaffInRequest,
        @Param('staffId') staffId: string,
        @Body(new JoiValidationPipe(updateStaffProfileSchema)) body: UpdateStaffProfileDto,
    ) {
        const { id: myStaffId, fullName } = staff;
        const response = await this.staffService.updateStaffProfile(staffId, body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'staff',
                id: response.id,
            },
            staffId: myStaffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-staff-profile',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STAFF_UPDATE')
    @Put(':staffId/email')
    @HttpCode(HttpStatus.OK)
    async updateStaffEmail(
        @Staff() staff: StaffInRequest,
        @Param('staffId') staffId: string,
        @Body(new JoiValidationPipe(updateStaffEmailSchema)) body: UpdateStaffEmailDto,
    ) {
        const { id: myStaffId, fullName } = staff;
        const response = await this.staffService.updateStaffEmail(staffId, body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'staff',
                id: response.id,
            },
            staffId: myStaffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-staff-email',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STAFF_UPDATE')
    @Put(':staffId/phone-number')
    @HttpCode(HttpStatus.OK)
    async updateStaffPhoneNumber(
        @Staff() staff: StaffInRequest,
        @Param('staffId') staffId: string,
        @Body(new JoiValidationPipe(updateStaffPhoneNumberSchema)) body: UpdateStaffPhoneNumberDto,
    ) {
        const { id: myStaffId, fullName } = staff;
        const response = await this.staffService.updateStaffPhoneNumber(staffId, body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'staff',
                id: response.id,
            },
            staffId: myStaffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-staff-phone-number',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STAFF_UPDATE')
    @Put(':staffId/password')
    @HttpCode(HttpStatus.OK)
    async updateStaffPassword(
        @Staff() staff: StaffInRequest,
        @Param('staffId') staffId: string,
        @Body(new JoiValidationPipe(updateStaffPasswordSchema)) body: UpdateStaffPasswordDto,
    ) {
        const { id: myStaffId, fullName } = staff;
        const { password } = body;

        const response = await this.staffService.updateStaffPassword(myStaffId, staffId, password);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'staff',
                id: response.id,
            },
            staffId: myStaffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-staff-password',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('STAFF_DELETE')
    @Delete(':staffId')
    @HttpCode(HttpStatus.OK)
    async deleteStaff(@Param('staffId') staffId: string, @Staff() staff: StaffInRequest) {
        const { id: myStaffId, fullName } = staff;

        const response = await this.staffService.deleteStaff(myStaffId, staffId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'staff',
                id: response.id,
            },
            staffId: myStaffId,
            executorName: fullName,
            action: 'DELETE',
            description: 'api-messages:delete-staff',
            data: response,
        });

        return response;
    }

    @Post('resend-verification-email')
    @HttpCode(HttpStatus.OK)
    async resendVerificationEmail(@Body(new JoiValidationPipe(staffEmailSchema)) body: StaffEmailDto) {
        await this.staffService.resendVerificationEmail(body.email);
        return {
            success: true,
            message: 'api-messages:verification-email-sent',
        };
    }
}
