import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import * as dayjs from 'dayjs';
import { EmailService } from 'src/email/email.service';
import { PrismaService } from 'src/prisma/prisma.service';
import { CustomException } from 'src/utils/CustomException';
import { v4 as uuidv4 } from 'uuid';

@Injectable()
export class StaffSharedService {
    constructor(
        private prisma: PrismaService,
        private emailService: EmailService,
    ) {}

    async emailCheck(
        email: string,
        excludedId?: string,
    ): Promise<{
        isCreated: boolean;
        isVerified: boolean;
    }> {
        const staff = await this.prisma.staff.findFirst({
            where: {
                email,
                deletedAt: null,
                id: !!excludedId
                    ? {
                          not: excludedId,
                      }
                    : undefined,
            },
        });

        if (!staff) {
            // Staff does not exist
            return {
                isCreated: false,
                isVerified: false,
            };
        }

        if (!staff.password) {
            // Staff exists but not verified
            return {
                isCreated: true,
                isVerified: false,
            };
        }

        // Staff exists and verified
        return {
            isCreated: true,
            isVerified: true,
        };
    }

    async sendVerificationEmail(staffId: string, email: string, fullName: string) {
        // Ensure the verification request is not more than 5 times per 12 hours
        const staffTokens = await this.prisma.staffToken.findMany({
            where: {
                staffId,
                type: 'CONFIRMATION',
                createdAt: {
                    gte: dayjs().subtract(12, 'hours').toDate(),
                },
            },
        });

        if (staffTokens.length >= 5) {
            throw new CustomException('api-messages:too-many-verification-requests', HttpStatus.TOO_MANY_REQUESTS, {
                tooManyRequests: true,
            });
        }

        // Create new token
        const newToken = uuidv4();
        const newTokenExpiredAt = dayjs().add(3, 'days');

        await this.prisma.staffToken.create({
            data: {
                staffId,
                token: newToken,
                type: 'CONFIRMATION',
                expiredAt: newTokenExpiredAt.toDate(),
            },
        });

        // Send verification email
        const emailResponse = await this.emailService.staffVerification(email, {
            name: fullName,
            staffId,
            token: newToken,
        });

        return emailResponse;
    }

    async resendVerificationEmail(email: string) {
        const { isCreated, isVerified } = await this.emailCheck(email);

        // Ensure staff is created but not yet verified
        if (!isCreated) {
            throw new CustomException('api-messages:staff-not-found', HttpStatus.NOT_FOUND, {
                isCreated,
            });
        }
        if (isVerified) {
            throw new CustomException('api-messages:staff-already-verified', HttpStatus.CONFLICT, {
                isCreated,
                isVerified,
            });
        }

        // Resend verification email
        const staff = await this.prisma.staff.findFirst({
            where: {
                email,
                deletedAt: null,
            },
            select: {
                id: true,
                email: true,
                fullName: true,
            },
        });

        const emailResponse = await this.sendVerificationEmail(staff.id, staff.email, staff.fullName);

        // If email failed to send, throw error
        if (!emailResponse.success) {
            throw new HttpException('api-messages:email-failed-to-send', HttpStatus.INTERNAL_SERVER_ERROR);
        }

        return staff;
    }
}
