import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { Book, Prisma } from '@prisma/client';
import * as dayjs from 'dayjs';
import { MediaService } from 'src/media/media.service';
import { PrismaService } from 'src/prisma/prisma.service';
import { UploadType } from 'src/types';
import { conditionalReturn } from 'src/utils';
import { AddBookDto, BookQuery, UpdateBookDto } from './book.dto';

@Injectable()
export class BookService {
    constructor(
        private prisma: PrismaService,
        private media: MediaService,
    ) {}

    async getBookByPagination(query: BookQuery) {
        const { page, pageSize, sortField, sortOrder, name, createdAt } = query;

        /* Where Options */
        const whereOptions: Prisma.BookWhereInput = {
            deletedAt: null,
            ...conditionalReturn(!!name, { name: { mode: 'insensitive', contains: name } }),
            ...conditionalReturn(createdAt && createdAt.length === 2, {
                createdAt: {
                    gte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[0]).startOf('day').toISOString()),
                    lte: createdAt && createdAt.length === 2 && new Date(dayjs(createdAt[1]).endOf('day').toISOString()),
                },
            }),
        };

        const bookCount = await this.prisma.book.count({
            where: whereOptions,
        });

        const currentPage = this.prisma.pageCounter(bookCount, page, pageSize);

        const bookList = await this.prisma.book.findMany({
            where: whereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            select: this.prisma.createSelect(['id', 'name', 'description', 'createdAt', 'price', 'availableForBuy', 'status']),
        });

        return {
            count: bookCount,
            rows: bookList,
            page: currentPage,
        };
    }

    async addBook(body: AddBookDto) {
        let book: Book;
        const { uploadType, uploadFile, bookImages, ...restBody } = body;

        switch (uploadType) {
            case UploadType.FILE:
                book = await this.prisma.book.create({
                    data: {
                        ...restBody,

                        bookMedias: {
                            createMany: {
                                data: uploadFile.map((data) => ({
                                    mediaId: data.mediaId,
                                })),
                            },
                        },

                        bookImages: {
                            createMany: {
                                data: bookImages.map((data) => ({
                                    mediaId: data.mediaId,
                                })),
                            },
                        },
                    },
                });
                break;
            case UploadType.LINK:
                book = await this.prisma.book.create({
                    data: {
                        ...restBody,
                        bookImages: {
                            createMany: {
                                data: bookImages.map((data) => ({
                                    mediaId: data.mediaId,
                                })),
                            },
                        },
                    },
                });
                break;
            default:
                book = await this.prisma.book.create({
                    data: {
                        ...restBody,
                        uploadType: UploadType.NONE,
                        bookImages: {
                            createMany: {
                                data: (bookImages as any[]).map((data) => ({
                                    mediaId: data.mediaId,
                                })),
                            },
                        },
                    },
                });
                break;
        }
        return book;
    }

    async getBookById(bookId: string) {
        const book = await this.prisma.book.findFirst({
            where: {
                id: bookId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.exclude('Book', ['deletedAt', 'updatedAt']),
                bookMedias: {
                    select: {
                        media: {
                            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
                        },
                    },
                },
                bookImages: {
                    select: {
                        mediaId: true,
                        media: {
                            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
                        },
                    },
                },
            },
        });

        if (!book) {
            throw new HttpException('api-messages:book-not-found', HttpStatus.NOT_FOUND);
        }

        return book;
    }

    async uploadBookFile(bookFile: Express.Multer.File) {
        const fileResponse = await this.media.upload(bookFile, { isPublic: false });

        const mediaResponse = await this.prisma.media.create({
            data: fileResponse,
            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
        });

        return mediaResponse;
    }

    async updateBookById(bookId: string, body: UpdateBookDto) {
        const book = await this.prisma.book.findFirst({
            where: {
                deletedAt: null,
                id: bookId,
            },
        });

        if (!book) {
            throw new HttpException('api-messages:book-not-found', HttpStatus.NOT_FOUND);
        }

        const { uploadType, uploadFile, bookImages, ...restBody } = body;
        let response;
        switch (body.uploadType) {
            case UploadType.FILE:
                response = await this.prisma.book.update({
                    where: {
                        id: bookId,
                        deletedAt: null,
                    },
                    data: {
                        ...restBody,
                        uploadType: UploadType.FILE,
                        bookMedias: {
                            deleteMany: {},
                            createMany: {
                                data: uploadFile.map((data) => ({
                                    mediaId: data.mediaId,
                                })),
                            },
                        },
                        bookImages: {
                            deleteMany: {},
                            createMany: {
                                data: bookImages.map((data) => ({
                                    mediaId: data.mediaId,
                                })),
                            },
                        },
                    },
                });
                break;
            case UploadType.LINK:
                response = await this.prisma.book.update({
                    where: {
                        id: bookId,
                        deletedAt: null,
                    },
                    data: {
                        ...restBody,
                        uploadType: UploadType.LINK,
                        bookImages: {
                            deleteMany: {},
                            createMany: {
                                data: bookImages.map((data) => ({
                                    mediaId: data.mediaId,
                                })),
                            },
                        },
                    },
                });
                break;
            case UploadType.NONE:
                response = await this.prisma.book.update({
                    where: {
                        id: bookId,
                        deletedAt: null,
                    },
                    data: {
                        bookMedias: {
                            deleteMany: {},
                        },
                        link: null,
                        ...restBody,
                        uploadType: {
                            set: UploadType.NONE,
                        },
                        bookImages: {
                            deleteMany: {},
                            createMany: {
                                data: bookImages.map((data) => ({
                                    mediaId: data.mediaId,
                                })),
                            },
                        },
                    },
                });
                break;
        }

        return response;
    }

    async updateBookStatus(bookId: string) {
        const book = await this.prisma.book.findFirst({
            where: {
                deletedAt: null,
                id: bookId,
            },
        });

        if (!book) {
            throw new HttpException('api-messages:book-not-found', HttpStatus.NOT_FOUND);
        }

        return await this.prisma.book.update({
            where: {
                id: bookId,
            },
            data: {
                status: book.status === 'ACTIVE' ? 'INACTIVE' : 'ACTIVE',
            },
            select: { id: true },
        });
    }

    async deleteBookById(bookId: string) {
        const book = await this.prisma.book.findFirst({
            where: {
                deletedAt: null,
                id: bookId,
            },
        });

        if (!book) {
            throw new HttpException('api-messages:book-not-found', HttpStatus.NOT_FOUND);
        }

        return await this.prisma.book.update({
            where: {
                id: bookId,
            },
            data: {
                deletedAt: new Date(),
            },
            select: { id: true },
        });
    }
}
