import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import { MediaService } from 'src/media/media.service';
import { PrismaService } from 'src/prisma/prisma.service';
import { BookOrderQuery, CreateBookOrderDto, UpdateBookOrderDto } from './book-order.dto';
import { Prisma } from '@prisma/client';
import { conditionalReturn } from 'src/utils';
import * as dayjs from 'dayjs';

@Injectable()
export class BookOrderService {
    constructor(
        private prisma: PrismaService,
        private media: MediaService,
    ) {}

    async getBookOrderListByPagination(query: BookOrderQuery) {
        const { page, pageSize, sortField, sortOrder, bookName, memberName, status } = query;

        /* Where Options */
        const whereOptions: Prisma.BookOrderWhereInput = {
            deletedAt: null,
            ...conditionalReturn(!!bookName, { book: { name: { mode: 'insensitive', contains: bookName } } }),
            ...conditionalReturn(!!memberName, { member: { fullName: { mode: 'insensitive', contains: memberName } } }),
            ...conditionalReturn(!!status, { status }),
        };

        const bookOrderCount = await this.prisma.bookOrder.count({
            where: whereOptions,
        });

        const currentPage = this.prisma.pageCounter(bookOrderCount, page, pageSize);

        const bookOrderList = await this.prisma.bookOrder.findMany({
            where: whereOptions,
            skip: (currentPage - 1) * pageSize,
            take: pageSize,
            orderBy: {
                [!sortField ? 'createdAt' : sortField]: sortOrder ?? 'asc',
            },
            select: this.prisma.createSelect(['id', 'book', 'member', 'status', 'createdAt', 'displayId', 'quantity', 'bookPrice']),
        });

        return {
            count: bookOrderCount,
            rows: bookOrderList,
            page: currentPage,
        };
    }

    async getAllBookOrder() {
        return await this.prisma.bookOrder.findMany({
            where: {
                deletedAt: null,
            },
            select: this.prisma.createSelect(['id', 'book', 'member', 'status', 'createdAt']),
        });
    }

    async getBookOrderById(bookOrderId: string) {
        const bookOrder = await this.prisma.bookOrder.findFirst({
            where: {
                id: bookOrderId,
                deletedAt: null,
            },
            select: {
                id: true,
                bookId: true,
                book: {
                    select: {
                        name: true,
                    },
                },
                memberId: true,
                member: {
                    select: {
                        fullName: true,
                    },
                },
                bookPrice: true,
                status: true,
                quantity: true,
                media: {
                    select: {
                        media: {
                            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
                        },
                    },
                },
            },
        });

        if (!bookOrder) {
            throw new HttpException('api-messages:book-order-not-found', HttpStatus.NOT_FOUND);
        }

        return bookOrder;
    }

    async createBookOrder(body: CreateBookOrderDto) {
        const { bookId, memberId, quantity } = body;

        const book = await this.prisma.book.findFirst({
            where: {
                id: bookId,
                deletedAt: null,
            },
        });

        if (!book) {
            throw new HttpException('api-messages:book-not-found', HttpStatus.NOT_FOUND);
        }

        const bookOrder = await this.prisma.bookOrder.create({
            data: {
                bookId,
                bookPrice: book.price,
                quantity,
                memberId,
            },
        });

        const updatedBookOrder = await this.updateDisplayId(bookOrder.id);

        return updatedBookOrder;
    }

    async updateBookOrder(bookOrderId: string, body: UpdateBookOrderDto) {
        const { quantity, bookPrice, status, media } = body;

        const bookOrder = await this.prisma.bookOrder.findFirst({
            where: {
                id: bookOrderId,
                deletedAt: null,
            },
        });

        if (!bookOrder) {
            throw new HttpException('api-messages:book-order-not-found', HttpStatus.NOT_FOUND);
        }

        const updatedBookOrder = await this.prisma.bookOrder.update({
            where: {
                id: bookOrder.id,
            },
            data: {
                quantity,
                bookPrice,
                status,
                media: {
                    deleteMany: {},
                    createMany: {
                        data: media.map((media) => ({
                            mediaId: media.mediaId,
                        })),
                    },
                },
            },
        });

        return updatedBookOrder;
    }

    async uploadBookOrderMedia(file: Express.Multer.File) {
        const fileResponse = await this.media.upload(file, { isPublic: false });

        const mediaResponse = await this.prisma.media.create({
            data: fileResponse,
            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
        });

        return mediaResponse;
    }

    async deleteBookOrder(bookOrderId: string) {
        const bookOrder = await this.prisma.bookOrder.findFirst({
            where: {
                id: bookOrderId,
                deletedAt: null,
            },
        });

        if (!bookOrder) {
            throw new HttpException('api-messages:book-order-not-found', HttpStatus.NOT_FOUND);
        }

        const deletedBookOrder = await this.prisma.bookOrder.update({
            where: {
                id: bookOrder.id,
            },
            data: {
                deletedAt: dayjs().toDate(),
            },
            select: {
                id: true,
            },
        });

        return deletedBookOrder;
    }

    private async updateDisplayId(bookOrderId: string) {
        const response = await this.prisma.bookOrder.findFirst({
            where: {
                id: bookOrderId,
            },
            select: {
                refNo: true,
            },
        });

        const ref = response.refNo.toString().padStart(2, '0');
        const displayId = `BO-${dayjs().format('YYMMDD')}-${ref}`;

        const updateResponse = await this.prisma.bookTokenSubscription.update({
            where: {
                id: bookOrderId,
            },
            data: {
                displayId,
            },
            select: {
                id: true,
            },
        });

        return updateResponse;
    }
}
