import { Body, Controller, Delete, Get, HttpCode, HttpStatus, Param, Post, Put, Query, UploadedFile, UseInterceptors } from '@nestjs/common';
import { ActivityLogService } from 'src/activity-log/activity-log.service';
import { BookOrderService } from './book-order.service';
import { Authenticator, Authorizer, Staff } from 'src/auth/auth.decorator';
import { StaffInRequest, User } from 'src/types';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { BookOrderQuery, CreateBookOrderDto, UpdateBookOrderDto } from './book-order.dto';
import { bookOrderQuerySchema, createBookOrderSchema, updateBookOrderSchema } from './book-order.validator';
import { FileInterceptor } from '@nestjs/platform-express';

@Controller()
export class BookOrderController {
    constructor(
        private bookOrderService: BookOrderService,
        private activityLogService: ActivityLogService,
    ) {}

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_ORDER_VIEW')
    @Get('')
    async getBookByPagination(@Query(new JoiValidationPipe(bookOrderQuerySchema)) query: BookOrderQuery) {
        const bookOrderList = await this.bookOrderService.getBookOrderListByPagination(query);

        return {
            total: bookOrderList.count,
            rows: bookOrderList.rows,
            page: bookOrderList.page,
        };
    }

    @Authenticator(User.STAFF)
    @Get('all')
    async getAllBookOrder() {
        const bookOrderList = await this.bookOrderService.getAllBookOrder();

        return bookOrderList;
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_ORDER_VIEW')
    @Get(':bookOrderId')
    async getBookOrderById(@Param('bookOrderId') bookOrderId: string) {
        return await this.bookOrderService.getBookOrderById(bookOrderId);
    }

    @Authenticator(User.STAFF)
    @Post('upload')
    @UseInterceptors(FileInterceptor('file'))
    async uploadBookOrderMedia(@UploadedFile() file: Express.Multer.File) {
        const response = await this.bookOrderService.uploadBookOrderMedia(file);
        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_ORDER_CREATE')
    @Post('')
    @HttpCode(HttpStatus.OK)
    async createBookOrder(@Staff() staff: StaffInRequest, @Body(new JoiValidationPipe(createBookOrderSchema)) body: CreateBookOrderDto) {
        const { id: staffId, fullName } = staff;
        const response = await this.bookOrderService.createBookOrder(body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'book-order',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'CREATE',
            description: 'api-messages:create-book-order',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_ORDER_UPDATE')
    @Put(':bookOrderId')
    @HttpCode(HttpStatus.OK)
    async updateBookOrder(
        @Staff() staff: StaffInRequest,
        @Param('bookOrderId') bookOrderId: string,
        @Body(new JoiValidationPipe(updateBookOrderSchema)) body: UpdateBookOrderDto,
    ) {
        const { id: staffId, fullName } = staff;
        const response = await this.bookOrderService.updateBookOrder(bookOrderId, body);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'book-order',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'UPDATE',
            description: 'api-messages:update-book-order',
            data: response,
        });

        return response;
    }

    @Authenticator(User.STAFF)
    @Authorizer('BOOK_ORDER_DELETE')
    @Delete(':bookOrderId')
    @HttpCode(HttpStatus.OK)
    async deleteBookOrder(@Staff() staff: StaffInRequest, @Param('bookOrderId') bookOrderId: string) {
        const { id: staffId, fullName } = staff;
        const response = await this.bookOrderService.deleteBookOrder(bookOrderId);

        await this.activityLogService.writeLogger({
            targetId: {
                tableName: 'book-order',
                id: response.id,
            },
            staffId,
            executorName: fullName,
            action: 'DELETE',
            description: 'api-messages:delete-book-order',
            data: response,
        });

        return response;
    }
}
