import { Controller, Get, HttpStatus, Query, HttpCode, Param, Put, Body, Post, UseInterceptors, UploadedFile } from '@nestjs/common';
import { StudyGroupService } from './study-group.service';
import { Authenticator, Member } from 'src/auth/auth.decorator';
import { MemberInRequest, Pagination, User } from 'src/types';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import {
    scheduleCommentSchema,
    studyGroupByIdSchema,
    studyGroupMemberCommentSchema,
    studyGroupQuerySchema,
    studyGroupTaskByIdSchema,
    updateCommentSchema,
    updateReplySchema,
    uploadMedicalCertificateSchema,
} from './study-group.validator';
import { StudyGroup } from 'src/types/study-group';
import { paginationSchema } from 'src/utils/validators/pagination.schema';
import { FileInterceptor } from '@nestjs/platform-express';
import { ScheduleComment, UpdateComment, UpdateReply, UploadMedicalCertificateDto } from './study-group.dto';
import { StudyGroupParams, StudyGroupTaskParams } from 'src/staff/study-group/study-group.dto';

@Controller()
export class StudyGroupController {
    constructor(private studyGroup: StudyGroupService) {}
    // Context Provider
    @Authenticator(User.MEMBER)
    @Get(':studyGroupId/id')
    @HttpCode(HttpStatus.OK)
    async getStudyGroupById(@Param(new JoiValidationPipe(studyGroupByIdSchema)) param: StudyGroupParams) {
        const studyGroup = await this.studyGroup.getStudyGroupById(param.studyGroupId);

        return studyGroup;
    }

    @Authenticator(User.MEMBER)
    @Get(':studyGroupId/member')
    @HttpCode(HttpStatus.OK)
    async getStudyGroupMembers(@Param(new JoiValidationPipe(studyGroupByIdSchema)) param: StudyGroupParams) {
        const studyGroupMembers = await this.studyGroup.getStudyGroupMembers(param.studyGroupId);

        return studyGroupMembers;
    }

    @Authenticator(User.MEMBER)
    @Get(':studyGroupTaskId/task-by-id')
    @HttpCode(HttpStatus.OK)
    async getStudyGroupTaskById(@Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams) {
        const studyGroupTask = await this.studyGroup.getStudyGroupTaskById(param.studyGroupTaskId);

        return studyGroupTask;
    }

    ///////////////////
    // Study Group
    ///////////////////

    // Get
    @Authenticator(User.MEMBER)
    @Get()
    @HttpCode(HttpStatus.OK)
    async getStudyGroupList(@Query(new JoiValidationPipe(studyGroupQuerySchema)) query: StudyGroup, @Member() member: MemberInRequest) {
        const studyGroupList = await this.studyGroup.getAllStudyGroupByPagination(query, member.id);

        return studyGroupList;
    }

    @Authenticator(User.MEMBER)
    @Get(':studyGroupId/member-penalty')
    @HttpCode(HttpStatus.OK)
    async getMemberPenalty(@Param(new JoiValidationPipe(studyGroupByIdSchema)) param: StudyGroupParams, @Member() member: MemberInRequest) {
        const memberPenalty = await this.studyGroup.getMemberPenalty(param.studyGroupId, member.id);

        return memberPenalty;
    }

    ///////////////////
    // Study Group Task
    ///////////////////

    // Get
    @Authenticator(User.MEMBER)
    @Get(':studyGroupTaskId/medical-certificate')
    @HttpCode(HttpStatus.OK)
    async getMedicalCertificate(
        @Member() member: MemberInRequest,
        @Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams,
    ) {
        const medicalCertificate = await this.studyGroup.getMedicalCertificate(param.studyGroupTaskId, member.id);

        return medicalCertificate;
    }

    // Create
    @Authenticator(User.MEMBER)
    @Post('upload')
    @UseInterceptors(FileInterceptor('file'))
    async uploadSubscriptionFile(@UploadedFile() file: Express.Multer.File) {
        const response = await this.studyGroup.uploadMedicalCertificateMedia(file);

        return response;
    }

    @Authenticator(User.MEMBER)
    @Post(':studyGroupTaskId')
    async uploadProofOfPayment(
        @Member() member: MemberInRequest,
        @Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams,
        @Body(new JoiValidationPipe(uploadMedicalCertificateSchema)) body: UploadMedicalCertificateDto,
    ) {
        const response = await this.studyGroup.uploadMedicalCertificate(member.id, param.studyGroupTaskId, body);

        return response;
    }

    ///////////////////
    // Comment
    ///////////////////

    // Get
    @Authenticator(User.MEMBER)
    @Get(':studyGroupId/member-comments')
    @HttpCode(HttpStatus.OK)
    async getMemberStudyGroupComments(
        @Param(new JoiValidationPipe(studyGroupByIdSchema)) param: StudyGroupParams,
        @Member() member: MemberInRequest,
    ) {
        const studyGroupTaskComment = await this.studyGroup.getMemberStudyGroupComments(member.id, param.studyGroupId);

        return studyGroupTaskComment;
    }

    @Authenticator(User.MEMBER)
    @Get(':studyGroupTaskId/task-comments')
    @HttpCode(HttpStatus.OK)
    async getStudyGroupTaskComment(
        @Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams,
        @Query(new JoiValidationPipe(paginationSchema)) query: Pagination,
        @Member() member: MemberInRequest,
    ) {
        const studyGroupTaskComment = await this.studyGroup.getTaskComments(param.studyGroupTaskId, member.id, query);

        return studyGroupTaskComment;
    }

    @Authenticator(User.MEMBER)
    @Get(':studyGroupTaskId/own-task-comment')
    @HttpCode(HttpStatus.OK)
    async getOwnTaskComment(@Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams, @Member() member: MemberInRequest) {
        const ownTaskComment = await this.studyGroup.getOwnTaskComment(param.studyGroupTaskId, member.id);

        return ownTaskComment;
    }

    @Authenticator(User.MEMBER)
    @Get('member-task-comments')
    @HttpCode(HttpStatus.OK)
    async getMemberTaskComments(@Member() member: MemberInRequest, @Query(new JoiValidationPipe(studyGroupMemberCommentSchema)) query: StudyGroup) {
        const memberTaskComments = await this.studyGroup.getMemberTaskComments(member.id, query);

        return memberTaskComments;
    }

    @Authenticator(User.MEMBER)
    @Get(':studyGroupTaskId/like')
    @HttpCode(HttpStatus.OK)
    async getTaskCommentLikes(@Param('studyGroupTaskId') studyGroupTaskId: string) {
        const taskCommentLikes = await this.studyGroup.getTaskCommentLikes(studyGroupTaskId);

        return taskCommentLikes;
    }

    @Authenticator(User.MEMBER)
    @Get(':studyGroupTaskId/reply/:commentId')
    @HttpCode(HttpStatus.OK)
    async getTaskCommentReplies(@Param('studyGroupTaskId') studyGroupTaskId: string, @Param('commentId') commentId: string) {
        const taskCommentReplies = await this.studyGroup.getReplies(studyGroupTaskId, commentId);

        return taskCommentReplies;
    }

    // Update
    @Authenticator(User.MEMBER)
    @Put(':studyGroupTaskId/comment')
    @HttpCode(HttpStatus.OK)
    async updateComment(
        @Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams,
        @Body(new JoiValidationPipe(updateCommentSchema)) body: UpdateComment,
        @Member() member: MemberInRequest,
    ) {
        const updatedComment = await this.studyGroup.updateComment(param.studyGroupTaskId, body?.comment, member?.id);
        return updatedComment;
    }

    @Authenticator(User.MEMBER)
    @Put(':studyGroupTaskId/reply')
    @HttpCode(HttpStatus.OK)
    async updateReply(
        @Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams,
        @Body(new JoiValidationPipe(updateReplySchema)) body: UpdateReply,
        @Member() member: MemberInRequest,
    ) {
        const updatedReply = await this.studyGroup.updateReply(param.studyGroupTaskId, member?.id, body?.commentId, body?.reply);
        return updatedReply;
    }

    @Authenticator(User.MEMBER)
    @Post(':studyGroupTaskId/schedule-comment')
    @HttpCode(HttpStatus.OK)
    async scheduleComment(
        @Param(new JoiValidationPipe(studyGroupTaskByIdSchema)) param: StudyGroupTaskParams,
        @Body(new JoiValidationPipe(scheduleCommentSchema)) body: ScheduleComment,
        @Member() member: MemberInRequest,
    ) {
        const scheduledComment = await this.studyGroup.scheduleComment(param.studyGroupTaskId, body?.comment, body?.prePlannedDate, member?.id);
        return scheduledComment;
    }

    @Authenticator(User.MEMBER)
    @Post(':studyGroupTaskId/like/:studyGroupTaskCommentId')
    @HttpCode(HttpStatus.OK)
    async likeTaskComment(
        @Param('studyGroupTaskId') studyGroupTaskId: string,
        @Member() member: MemberInRequest,
        @Param('studyGroupTaskCommentId') studyGroupTaskCommentId: string,
    ) {
        const likedComment = await this.studyGroup.likeTaskComment(studyGroupTaskCommentId, member.id, studyGroupTaskId);

        return likedComment;
    }
}
