import { HttpException, HttpStatus, Injectable } from '@nestjs/common';
import * as argon from 'argon2';
import memberData from 'src/data/member';
import { PrismaService } from 'src/prisma/prisma.service';
import { UpdateMyPasswordDto, UpdateMyPhoneNumberDto, UpdateMyProfileDto } from './my-profile.dto';

@Injectable()
export class MemberMyProfileService {
    constructor(private readonly prisma: PrismaService) {}

    async getMyProfile(memberId: string) {
        const member = await this.prisma.member.findUnique({
            where: {
                id: memberId,
            },
            select: {
                ...this.prisma.createSelect(memberData.exclude(['password', 'deletedAt'])),
            },
        });

        return member;
    }

    async updateMyProfile(memberId: string, body: UpdateMyProfileDto) {
        const { fullName, preferredName, address, dateOfBirth, source, preferredLanguage, country, state, city } = body;

        const member = await this.prisma.member.findFirst({
            where: {
                id: memberId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        const updatedMember = await this.prisma.member.update({
            where: {
                id: member.id,
            },
            data: {
                fullName,
                preferredName,
                address,
                dateOfBirth,
                source,
                preferredLanguage,
                country: city === '' ? country : 'Malaysia',
                state,
                city,
            },
            select: this.prisma.createSelect(['id']),
        });

        return updatedMember;
    }

    async updateMyPhoneNumber(memberId: string, body: UpdateMyPhoneNumberDto) {
        const { phoneNumber } = body;

        const member = await this.prisma.member.findFirst({
            where: {
                id: memberId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        const updatedMember = await this.prisma.member.update({
            where: {
                id: member.id,
            },
            data: {
                phoneNumber,
            },
            select: this.prisma.createSelect(['id']),
        });

        return updatedMember;
    }

    async updateMyPassword(memberId: string, body: UpdateMyPasswordDto) {
        const { password } = body;

        const member = await this.prisma.member.findFirst({
            where: {
                id: memberId,
                deletedAt: null,
            },
            select: {
                ...this.prisma.createSelect(['id']),
            },
        });

        if (!member) {
            throw new HttpException('api-messages:member-not-found', HttpStatus.NOT_FOUND);
        }

        // Hash password
        const hashedPassword = await argon.hash(password);

        const updatedMember = await this.prisma.member.update({
            where: {
                id: member.id,
            },
            data: {
                password: hashedPassword,
            },
            select: this.prisma.createSelect(['id']),
        });

        return updatedMember;
    }
}
