import { Body, HttpStatus, Injectable } from '@nestjs/common';
import { EmailService } from 'src/email/email.service';
import { PrismaService } from 'src/prisma/prisma.service';
import { MemberSharedService } from '../member-shared.service';
import { CustomException } from 'src/utils/CustomException';
import * as dayjs from 'dayjs';
import { Member } from '@prisma/client';
import { v4 as uuidv4 } from 'uuid';

@Injectable()
export class MemberService {
    constructor(
        private prisma: PrismaService,
        private emailService: EmailService,
        private sharedService: MemberSharedService,
    ) {}

    async registerMember(@Body() body: Member) {
        // Check existing email
        const { isCreated } = await this.sharedService.emailCheck(body.email);

        // If email already exists, throw error
        if (isCreated) {
            throw new CustomException('api-messages:email-already-exists', HttpStatus.BAD_REQUEST, {
                isCreated: true,
            });
        }

        // Create member
        const member = await this.prisma.member.create({
            data: {
                email: body.email,
                fullName: body.fullName,
                preferredName: body.preferredName,
                phoneNumber: body.phoneNumber,
                address: body.address,
                dateOfBirth: body.dateOfBirth,
                source: body.source,
                country: body.country,
                state: body.state,
                city: body.city,
            },
            select: {
                ...this.prisma.createSelect(['id', 'email', 'fullName', 'preferredName', 'phoneNumber', 'address', 'dateOfBirth']),
            },
        });

        // Send verification email
        const verificationEmail = await this.sendVerificationEmail(member.id, member.email, member.fullName);

        if (!verificationEmail) {
            throw new CustomException('api-messages:email-verification-failed', HttpStatus.INTERNAL_SERVER_ERROR, {
                emailVerificationFailed: true,
            });
        }

        return member;
    }

    async sendVerificationEmail(memberId: string, email: string, fullName: string) {
        // Ensure the verification request is not more than 5 times per 12 hours
        const memberTokens = await this.prisma.memberToken.findMany({
            where: {
                memberId,
                type: 'CONFIRMATION',
                createdAt: {
                    gte: dayjs().subtract(12, 'hours').toDate(),
                },
            },
        });

        if (memberTokens.length >= 5) {
            throw new CustomException('api-messages:too-many-verification-requests', HttpStatus.TOO_MANY_REQUESTS, {
                tooManyRequests: true,
            });
        }

        // Create new token
        const newToken = uuidv4();
        const newTokenExpiredAt = dayjs().add(3, 'days');

        await this.prisma.memberToken.create({
            data: {
                memberId,
                token: newToken,
                type: 'CONFIRMATION',
                expiredAt: newTokenExpiredAt.toDate(),
            },
        });

        // Send verification email
        const emailResponse = await this.emailService.memberVerification(email, {
            name: fullName,
            memberId,
            token: newToken,
        });

        return emailResponse;
    }
}
