import { Controller, Get, Param, Put, Query } from '@nestjs/common';
import { BookService } from './book.service';
import { Authenticator, Member } from 'src/auth/auth.decorator';
import { MemberInRequest, User } from 'src/types';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { bookIdParamSchema, bookQuerySchema } from './book.validator';
import { BookQuery } from 'src/staff/books/book.dto';
import { BookIdParam } from './book.dto';

@Controller()
export class BookController {
    constructor(private bookService: BookService) {}

    @Authenticator(User.MEMBER)
    @Get('')
    async getBookListByPagination(
        @Query(new JoiValidationPipe(bookQuerySchema))
        query: BookQuery,
    ) {
        const bookList = await this.bookService.getBookByPagination(query);

        return {
            total: bookList.count,
            rows: bookList.rows,
            page: bookList.page,
        };
    }

    @Authenticator(User.MEMBER)
    @Get('request')
    async getBookRequestListByPagination(
        @Query(new JoiValidationPipe(bookQuerySchema))
        query: BookQuery,
        @Member() member: MemberInRequest,
    ) {
        const bookList = await this.bookService.getBookRequestByPagination(query, member.id);

        return {
            total: bookList.count,
            rows: bookList.rows,
            page: bookList.page,
        };
    }

    @Authenticator(User.MEMBER)
    @Get(':bookId')
    async getBookById(@Param(new JoiValidationPipe(bookIdParamSchema)) param: BookIdParam) {
        const response = await this.bookService.getBookById(param.bookId);

        return response;
    }

    @Authenticator(User.MEMBER)
    @Put(':bookId/request')
    async requestBook(@Param('bookId') bookId: string, @Member() member: MemberInRequest) {
        const response = await this.bookService.requestBook(bookId, member.id);

        return response;
    }

    @Authenticator(User.MEMBER)
    @Put(':bookId/request/:requestId/cancel-request')
    async cancelBookRequest(@Param('bookId') bookId: string, @Param('requestId') requestId: string, @Member() member: MemberInRequest) {
        const response = await this.bookService.cancelBookRequest(bookId, requestId, member.id);

        return response;
    }
}
