import { Body, Controller, Delete, Get, Param, Post, Query, UploadedFile, UseInterceptors } from '@nestjs/common';
import { MemberBookTokenService } from './book-token.service';
import { JoiValidationPipe } from 'src/utils/validators/joi.validation.pipe';
import { paginationSchema } from 'src/utils/validators/pagination.schema';
import { MemberInRequest, Pagination, User } from 'src/types';
import { Authenticator, Member } from 'src/auth/auth.decorator';
import { FileInterceptor } from '@nestjs/platform-express';
import { BookTokenIdParam, BookTokenSubscriptionIdParam, RequestBookTokenDto, UploadProofOfPaymentDto } from './book-token.dto';
import { bookTokenIdSchema, bookTokenSubscriptionIdSchema, requestBookTokenSchema, uploadProofOfPaymentSchema } from './book-token.validator';

@Controller()
export class MemberBookTokenController {
    constructor(private readonly bookTokenService: MemberBookTokenService) {}

    @Authenticator(User.MEMBER)
    @Get()
    async getBookTokenSubscriptionList(@Query(new JoiValidationPipe(paginationSchema)) query: Pagination, @Member() member: MemberInRequest) {
        const response = this.bookTokenService.getBookTokenSubscriptionList(member.id, query);

        return response;
    }

    @Authenticator(User.MEMBER)
    @Get('/:bookTokenId')
    async getBookTokenSubscriptionById(@Param(new JoiValidationPipe(bookTokenIdSchema)) param: BookTokenIdParam) {
        const response = this.bookTokenService.getBookTokenSubscriptionById(param.bookTokenId);
        return response;
    }

    @Authenticator(User.MEMBER)
    @Post()
    async requestBookToken(@Member() member: MemberInRequest, @Body(new JoiValidationPipe(requestBookTokenSchema)) body: RequestBookTokenDto) {
        const response = this.bookTokenService.requestBookToken(member.id, body);
        return response;
    }

    @Authenticator(User.MEMBER)
    @Delete(':bookTokenSubscriptionId')
    async cancelBookTokenSubscription(
        @Member() member: MemberInRequest,
        @Param(new JoiValidationPipe(bookTokenSubscriptionIdSchema)) param: BookTokenSubscriptionIdParam,
    ) {
        const response = this.bookTokenService.cancelBookTokenRequest(member.id, param.bookTokenSubscriptionId);

        return response;
    }

    @Authenticator(User.MEMBER)
    @Post('upload')
    @UseInterceptors(FileInterceptor('file'))
    async uploadSubscriptionFile(@UploadedFile() file: Express.Multer.File) {
        const response = await this.bookTokenService.uploadPaymentProofMedia(file);

        return response;
    }

    @Authenticator(User.MEMBER)
    @Post(':bookTokenSubscriptionId')
    async uploadProofOfPayment(
        @Member() member: MemberInRequest,
        @Param(new JoiValidationPipe(bookTokenSubscriptionIdSchema)) param: BookTokenSubscriptionIdParam,
        @Body(new JoiValidationPipe(uploadProofOfPaymentSchema)) body: UploadProofOfPaymentDto,
    ) {
        const response = await this.bookTokenService.uploadProofOfPayment(member.id, param.bookTokenSubscriptionId, body);

        return response;
    }
}
