import { Injectable } from '@nestjs/common';
import * as convert from 'heic-convert';
import * as mime from 'mime'; //Use version 3.0.0 
import * as sharp from 'sharp';
import { IMAGE_QUALITY_REDUCED_BY_PERCENTAGE } from 'src/config';
import { PrismaService } from 'src/prisma/prisma.service';
import { Media, UploadOptions } from 'src/types';
import { S3Service } from './s3.service';

@Injectable()
export class MediaService {
    constructor(
        private s3: S3Service,
        private prisma: PrismaService,
    ) {}

    async upload(media: Media, options: UploadOptions) {
        const key = await this.s3.uploadFile(media);
        const name = Buffer.from(media.originalname, 'latin1').toString('utf8');

        return {
            name,
            type: mime.getType(media.originalname.split('.')[media.originalname.split('.').length - 1]) ?? '',
            size: media.size,
            key: key,
            isPublic: options.isPublic ?? false,
            referenceTable: options.referenceTable ?? undefined,
            referenceId: options.referenceId ?? undefined,
            permission: options.permission ?? undefined,
            uploadedBy: options.uploadedBy ?? undefined,
            userId: options.userId ?? undefined,
        };
    }

    async get(key: string) {
        const fileType = key.split('.')[1];
        const fileStream = await this.s3.getFileStream(key);

        if (!fileStream) {
            throw {
                status: 404,
                message: 'api-messages:media-not-found',
            };
        }

        // Get from database
        const media = await this.prisma.media.findFirst({
            where: {
                key,
                deletedAt: null,
            },
            select: this.prisma.createSelect(['id', 'name', 'type', 'key']),
        });

        if (!media) {
            throw {
                status: 404,
                message: 'api-messages:media-not-found',
            };
        }
        // If file is not image or pdf, return fileStream with attachment true
        if (mime.getType(fileType) !== 'undefined' && (mime.getType(fileType)!.split('/')[0] === 'image' || fileType === 'pdf')) {
            return {
                attachment: true,
                fileStream,
                media,
            };
        }

        return {
            fileStream,
            media,
        };
    }

    async convertHeic(media: Express.Multer.File, format: 'JPEG' | 'PNG'): Promise<Express.Multer.File> {
        const convertedImage = await convert({ buffer: media.buffer, format, quality: 1 });
        const bufferImage = Buffer.from(convertedImage);

        return {
            ...media,
            buffer: bufferImage,
            size: bufferImage.byteLength,
            mimetype: `image/${format.toLowerCase()}`,
            originalname: `${media.originalname.split('.')[0]}.${format.toLowerCase()}`,
        };
    }

    async sharpenImage(media: Express.Multer.File): Promise<Express.Multer.File> {
        const { format } = await sharp(media.buffer).metadata();

        let imageSharp: Buffer;
        let image = media;
        switch (format) {
            case 'jpeg':
                imageSharp = await sharp(image.buffer)
                    .jpeg({ quality: 100 - IMAGE_QUALITY_REDUCED_BY_PERCENTAGE })
                    .toBuffer();
                break;
            case 'png':
                imageSharp = await sharp(image.buffer)
                    .png({ quality: 100 - IMAGE_QUALITY_REDUCED_BY_PERCENTAGE })
                    .toBuffer();
                break;
            case 'webp':
                imageSharp = await sharp(image.buffer)
                    .webp({ quality: 100 - IMAGE_QUALITY_REDUCED_BY_PERCENTAGE })
                    .toBuffer();
                break;
            case 'heif':
                image = await this.convertHeic(image, 'JPEG');
                imageSharp = await sharp(image.buffer)
                    .jpeg({ quality: 100 - IMAGE_QUALITY_REDUCED_BY_PERCENTAGE })
                    .toBuffer();
                break;
            default:
                imageSharp = image.buffer;
        }

        return {
            ...image,
            buffer: imageSharp,
        };
    }

    async resizeImage(image: Express.Multer.File, size: [number, number]): Promise<Express.Multer.File> {
        const thumbnail = await sharp(image.buffer).resize(size[0], size[1], { fit: 'inside' }).toBuffer();

        return {
            ...image,
            buffer: thumbnail,
            size: thumbnail.byteLength,
        };
    }
}
