import { GetServerSideProps, NextPage } from 'next';
import { useTranslation } from 'next-i18next';
import { serverSideTranslations } from 'next-i18next/serverSideTranslations';
import Layout from '@/components/layouts';
import { BasePageProps } from '@/types';
import { authentication } from '@/utils/authentication';
import localeLinkGenerator from '@/utils/localeLinkGenerator';
import StatisticCard from '@/components/dashboard/StatisticCard';
import { Empty, Spin } from 'antd';
import { HiMiniUserGroup } from 'react-icons/hi2';
import { FileSearchOutlined } from '@ant-design/icons';
import { AiOutlineExclamationCircle } from 'react-icons/ai';
import { useQuery } from '@tanstack/react-query';
import { getDashboardStatistics } from '@/services/dashboard';

const Dashboard: NextPage<BasePageProps> = ({ staff }) => {
    const { t } = useTranslation(['dashboard', 'layout', 'common', 'messages']);

    const { isFetching, data } = useQuery({
        queryFn: async () => {
            const response = await getDashboardStatistics();

            return response.data;
        },
    });

    const seoConfig = {
        title: t('dashboard'),
    };

    const breadCrumbItems = [
        {
            label: t('layout:dashboard'),
            path: '/',
        },
    ];

    return (
        <Layout staff={staff} breadCrumbItems={breadCrumbItems} activeMenu={['dashboard']} seoConfig={seoConfig}>
            {isFetching ? (
                <Spin spinning />
            ) : !data ? (
                <Empty />
            ) : (
                <>
                    <div className="flex flex-col lg:flex-row gap-4 mb-4">
                        <StatisticCard title={t('number-of-members')} icon={<HiMiniUserGroup />} loading={isFetching} value={data.numberOfMembers} />
                        <StatisticCard
                            title={t('number-of-study-request')}
                            icon={<FileSearchOutlined />}
                            loading={isFetching}
                            value={data.numberOfStudyRequest}
                        />
                        <StatisticCard
                            title={t('number-of-penalties-this-week')}
                            icon={<AiOutlineExclamationCircle />}
                            loading={isFetching}
                            value={data.numberOfPenaltiesThisWeek}
                        />
                    </div>
                </>
            )}
        </Layout>
    );
};

export const getServerSideProps: GetServerSideProps = async ({ req, locale, resolvedUrl }) => {
    try {
        const authResponse = await authentication(req);

        return {
            props: {
                staff: authResponse,
                ...(await serverSideTranslations(locale as string, ['dashboard', 'layout', 'common', 'messages', 'api-messages'])),
            },
        };
    } catch (error: any) {
        if (error.response?.data?.unauthorized) {
            return {
                redirect: {
                    destination: localeLinkGenerator(locale, `/unauthorized`),
                    permanent: false,
                },
            };
        }

        return {
            redirect: {
                destination: localeLinkGenerator(locale, `/?redirect=/${resolvedUrl}`),
                permanent: false,
            },
        };
    }
};

export default Dashboard;
