import useIsMounted from '@/hooks/useIsMounted';
import '@/styles/globals.css';
import { QueryClient, QueryClientProvider } from '@tanstack/react-query';
import { ReactQueryDevtools } from '@tanstack/react-query-devtools';
import { ConfigProvider, ConfigProviderProps } from 'antd';
import dayjs from 'dayjs';
import type { AppProps } from 'next/app';
import { Poppins } from 'next/font/google';
import { useRouter } from 'next/router';
import NextNProgress from 'nextjs-progressbar';
import { DefaultSeo } from 'next-seo';
import { ToastContainer } from 'react-toastify';
import { APP_NAME, SEO_TITLE, TOAST_DURATION } from '@/configs';
import en_GB from 'antd/locale/en_GB';
import zh_CN from 'antd/locale/zh_CN';
import ms_MY from 'antd/locale/ms_MY';
import Initialize from '@/components/globals/Initialize';
import StaffProvider from '@/providers/StaffContext';
import { appWithTranslation, useTranslation } from 'next-i18next';
import { Locales } from '@/types';
import 'react-toastify/dist/ReactToastify.css';
import utc from 'dayjs/plugin/utc';
import timezone from 'dayjs/plugin/timezone';

const poppins = Poppins({
    weight: ['400', '500', '600', '700'],
    subsets: ['latin'],
    display: 'swap',
    variable: '--font-poppins',
});

const queryClient = new QueryClient({
    defaultOptions: {
        queries: {
            retry: false,
            refetchOnWindowFocus: false,
        },
    },
});

const App = ({ Component, pageProps }: AppProps) => {
    const { t } = useTranslation(['messages']);
    const router = useRouter();
    const isMounted = useIsMounted();
    dayjs.locale(router.locale?.toLowerCase() || 'en-gb');
    dayjs.extend(utc);
    dayjs.extend(timezone);
    dayjs.tz.setDefault('Asia/Kuala_Lumpur');
    const antdGlobalLocale = () => {
        switch (router.locale as Locales) {
            case 'zh-CN':
                return zh_CN;
            case 'ms-MY':
                return ms_MY;
            default:
                return en_GB;
        }
    };

    const antDesignGlobalConfig: ConfigProviderProps = {
        theme: {
            token: {
                fontFamily: poppins.style.fontFamily,
            },
        },
        autoInsertSpaceInButton: false,
        locale: antdGlobalLocale(),
        form: {
            validateMessages: {
                required: t('required', { name: '${label}' }),
                types: {
                    email: t('invalid', { name: '${label}' }),
                },
            },
            scrollToFirstError: true,
        },
    };
    return (
        <StaffProvider>
            <QueryClientProvider client={queryClient}>
                <ConfigProvider {...antDesignGlobalConfig}>
                    <DefaultSeo
                        title={undefined}
                        defaultTitle={APP_NAME}
                        titleTemplate={`%s | ${SEO_TITLE}`}
                        description={SEO_TITLE}
                        additionalLinkTags={[
                            {
                                rel: 'icon',
                                href: '/images/logo_cropped_noBg.png?',
                            },
                        ]}
                    />
                    <NextNProgress />
                    {isMounted ? (
                        <main className={`${poppins.variable} ${poppins.className} font-roboto`}>
                            <Component {...pageProps} />
                            <ToastContainer autoClose={TOAST_DURATION} closeOnClick={false} />
                        </main>
                    ) : (
                        <Initialize />
                    )}
                </ConfigProvider>
                <ReactQueryDevtools position="bottom-right" />
            </QueryClientProvider>
        </StaffProvider>
    );
};

export default appWithTranslation(App);
