import { CloudDownloadOutlined } from '@ant-design/icons';
import { Button } from 'antd';
import { useTranslation } from 'next-i18next';
import ExcelJS from 'exceljs';
import { useQuery, useQueryClient } from '@tanstack/react-query';
import { getExportStudyGroupMemberList } from '@/services/study-group';
import { AxiosErrorResponse, StudyGroupMembers } from '@/types';
import { toast } from 'react-toastify';
import { errorMessageFormatter } from '@/utils';
import { dateTransformer } from '@/utils/timeTransformer';
import { useRouter } from 'next/router';

// PDF
interface ExportDataProps {
    fileName: string;
    query: StudyGroupMembers;
}

const ExportData: React.FC<ExportDataProps> = ({ fileName, query }) => {
    const { t } = useTranslation(['study-group', 'layout', 'common', 'messages']);
    const router = useRouter();
    const { studyGroupId } = router.query;
    const queryClient = useQueryClient();

    useQuery({
        queryKey: ['book-token-request', 'export'],
        enabled: false,
        queryFn: async () => {
            const res = await getExportStudyGroupMemberList(query, studyGroupId as string);

            return res.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    const workbook = new ExcelJS.Workbook();
    const worksheet = workbook.addWorksheet('Study Group Task Comments');

    // Columns
    worksheet.columns = [
        { header: 'Full Name', key: 'fullName', width: 20 },
        { header: 'Preferred Name', key: 'preferredName', width: 20 },
        { header: 'Phone Number', key: 'phoneNumber', width: 20 },
        { header: 'Email', key: 'email', width: 20 },
        { header: 'Created At', key: 'createdAt', width: 20 },
    ];

    // Add Row Style
    worksheet.getRow(1).font = { bold: true };
    worksheet.getRow(1).alignment = { horizontal: 'center' };
    worksheet.getRow(1).fill = {
        type: 'pattern',
        pattern: 'solid',
        fgColor: { argb: 'FFA9A9A9' },
    };

    const generateExcelHandler = async (data: any) => {
        data?.map((member: any) => {
            const extractMember = member.member;

            worksheet.addRow({
                fullName: extractMember.fullName,
                preferredName: extractMember.preferredName || '-',
                phoneNumber: `;${extractMember.phoneNumber}`,
                email: extractMember.email,
                createdAt: dateTransformer(extractMember.createdAt),
            });
        });

        // Generate the Excel file
        await workbook.xlsx.writeBuffer().then((buffer: any) => {
            const blob = new Blob([buffer], { type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' });
            const link = document.createElement('a');
            const url = URL.createObjectURL(blob);
            link.href = url;
            link.setAttribute('download', fileName);
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
        });
    };

    const exportToCsvHandler = () => {
        queryClient.fetchQuery(['book-token-request', 'export']).then((res) => {
            generateExcelHandler(res);
        });
    };

    return (
        <Button type="primary" icon={<CloudDownloadOutlined />} onClick={exportToCsvHandler}>
            {t('export-csv')}
        </Button>
    );
};

export default ExportData;
