import { useEffect, useState } from 'react';
import { MdOutlineEdit } from 'react-icons/md';
import { useTranslation } from 'next-i18next';
import { UseQueryResult } from '@tanstack/react-query';
import { Button, Input, Spin, Typography } from 'antd';
import EmailModal from '@/components/staff/modals/EmailModal';
import { useStaffContext } from '@/providers/StaffContext';
import { Staff } from '@/types';
import PasswordModal from '../modals/PasswordModal';
import PhoneNumberModal from '../modals/PhoneModal';

interface CredentialTabProps {
    staffId: string;
    staffQuery: UseQueryResult<Staff | null>;
}

const Credential: React.FC<CredentialTabProps> = ({ staffQuery }) => {
    const { t } = useTranslation(['staff', 'common']);
    const { staff } = useStaffContext();
    const staffToBeEdit = staffQuery.data;
    const { Title } = Typography;

    const [email, setEmail] = useState<string>('');
    const [phoneNumber, setPhoneNumber] = useState<string>('');
    const [isEmailModalOpen, setIsEmailModalOpen] = useState<boolean>(false);
    const [isPhoneNumberModalOpen, setIsPhoneNumberModalOpen] = useState<boolean>(false);
    const [isPasswordModalOpen, setIsPasswordModalOpen] = useState<boolean>(false);

    useEffect(() => {
        if (staffToBeEdit) {
            setEmail(staffToBeEdit.email);
            setPhoneNumber(staffToBeEdit.phoneNumber);
        }
    }, [staffToBeEdit]);

    return (
        <Spin spinning={staffQuery.isLoading}>
            {staffToBeEdit && (
                <div>
                    {/* Email Input */}
                    <div className="mb-6">
                        <Title level={5}>{t('email')}</Title>
                        <div className="flex gap-2 items-center">
                            <Input name="email" disabled value={email} className="max-w-[300px]" />
                            {staff?.role.STAFF_UPDATE && (
                                <Button
                                    type="primary"
                                    onClick={() => {
                                        setIsEmailModalOpen(true);
                                    }}
                                >
                                    <MdOutlineEdit className="text-xl" />
                                </Button>
                            )}
                        </div>
                    </div>
                    {/* Phone Number Input */}
                    <div className="mb-6">
                        <Title level={5}>{t('phone-number')}</Title>
                        <div className="flex gap-2 items-center">
                            <Input name="phoneNumber" disabled value={phoneNumber} className="max-w-[300px]" />
                            {staff?.role.STAFF_UPDATE && (
                                <Button
                                    type="primary"
                                    onClick={() => {
                                        setIsPhoneNumberModalOpen(true);
                                    }}
                                >
                                    <MdOutlineEdit className="text-xl" />
                                </Button>
                            )}
                        </div>
                    </div>
                    {/* Password Input */}
                    <div className="mb-4">
                        <Title level={5}>{t('password')}</Title>
                        <div className="flex gap-2 items-center">
                            <Input.Password disabled defaultValue="************" className="max-w-[300px]" visibilityToggle={false} />
                            <Button type="primary" onClick={() => setIsPasswordModalOpen(true)}>
                                <MdOutlineEdit className="text-xl" />
                            </Button>
                        </div>
                    </div>
                    <div>
                        <EmailModal open={isEmailModalOpen} setOpen={setIsEmailModalOpen} staff={staffToBeEdit} staffQuery={staffQuery} />
                        <PhoneNumberModal
                            open={isPhoneNumberModalOpen}
                            setOpen={setIsPhoneNumberModalOpen}
                            staff={staffToBeEdit}
                            staffQuery={staffQuery}
                        />
                        <PasswordModal open={isPasswordModalOpen} setOpen={setIsPasswordModalOpen} staff={staffToBeEdit} staffQuery={staffQuery} />
                    </div>
                </div>
            )}
        </Spin>
    );
};

export default Credential;
