import { useState } from 'react';
import 'react-phone-input-2/lib/style.css';
import { toast } from 'react-toastify';
import { useTranslation } from 'next-i18next';
import { UseQueryResult, useMutation } from '@tanstack/react-query';
import { Button, Form, Input, Modal } from 'antd';
import PasswordHintContainer from '@/components/ui/input/password/PasswordHintContainer';
import { updateStaffPassword } from '@/services/staff';
import { BaseModalProps, Staff, UpdateStaffPasswordParams } from '@/types';
import { errorMessageFormatter } from '@/utils/errorFormatter';
import { passwordRegex } from '@/utils/regex';

interface PasswordModalProps extends BaseModalProps {
    staff: Staff;
    staffQuery: UseQueryResult<Staff | null>;
}

const PasswordModal: React.FC<PasswordModalProps> = ({ open, setOpen, staff, staffQuery }) => {
    const { t } = useTranslation(['staff', 'common']);
    const [form] = Form.useForm<{ password: string }>();
    const [password, setPassword] = useState<string>('');

    // Mutations
    const updateStaffPasswordMutation = useMutation({
        mutationFn: async ({ staffId, body }: { staffId: string; body: UpdateStaffPasswordParams }) => {
            const res = await updateStaffPassword(staffId, { password: body.password });
            return res.data;
        },
        onSuccess: () => {
            form.resetFields();
            setOpen(false);
            staffQuery.refetch();
        },
    });

    // Functions
    const onCancelHandler = () => {
        form.resetFields();
        setOpen(false);
    };

    const onUpdateStaffPasswordHandler = () => {
        form.validateFields().then(async (values) => {
            toast.promise(updateStaffPasswordMutation.mutateAsync({ staffId: staff.id, body: values }), {
                pending: t('messages:updating-staff-password'),
                success: t('messages:staff-password-updated'),
                error: {
                    render({ data }: any) {
                        return t(errorMessageFormatter(data));
                    },
                },
            });
        });
    };

    const handlePasswordOnChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        e.preventDefault();
        setPassword(e.target.value);
    };

    return (
        <Modal open={open} onCancel={onCancelHandler} footer={null} title={t('edit-password')}>
            <Form layout="vertical" form={form} className="mt-4" size="large">
                <Form.Item
                    name="password"
                    label={t('password')}
                    rules={[
                        { required: true },
                        {
                            pattern: passwordRegex,
                            message: '',
                        },
                    ]}
                    className="mb-5"
                    labelCol={{ flex: '35px' }}
                >
                    <Input.Password classNames={{ prefix: '!me-2' }} onChange={handlePasswordOnChange} />
                </Form.Item>
                <div className="mt-5">
                    <PasswordHintContainer password={password} />
                </div>
                <Form.Item
                    name="confirmPassword"
                    label={t('confirm-password')}
                    dependencies={['password']}
                    rules={[
                        { required: true },
                        {
                            pattern: passwordRegex,
                            message: '',
                        },
                        {
                            validator: (_, value) => (value === password ? Promise.resolve() : Promise.reject(t('messages:password-not-match'))),
                        },
                    ]}
                    labelCol={{ flex: '35px' }}
                >
                    <Input.Password classNames={{ prefix: '!me-2' }} />
                </Form.Item>
                <div className="pt-2 flex gap-3">
                    <Button loading={updateStaffPasswordMutation.isLoading} size="large" block onClick={onCancelHandler}>
                        {t('common:cancel')}
                    </Button>
                    <Button loading={updateStaffPasswordMutation.isLoading} size="large" type="primary" block onClick={onUpdateStaffPasswordHandler}>
                        {t('common:update')}
                    </Button>
                </div>
            </Form>
        </Modal>
    );
};

export default PasswordModal;
