import Dragger from '@/components/Dragger';
import { getNotice, updateNotice } from '@/services/site-setting';
import { AxiosErrorResponse } from '@/types';
import { errorMessageFormatter } from '@/utils';
import { ArrowDownOutlined, ArrowUpOutlined, DeleteOutlined } from '@ant-design/icons';
import { useMutation, useQuery } from '@tanstack/react-query';
import { Editor } from '@tinymce/tinymce-react';
import { Button, Card, Form } from 'antd';
import { useTranslation } from 'next-i18next';
import { toast } from 'react-toastify';
import { v4 as uuid } from 'uuid';

const NoticeTab: React.FC = () => {
    const { t, i18n } = useTranslation(['site-setting', 'layout', 'common', 'messages']);
    const currentLocale = i18n.language;

    const [noticeForm] = Form.useForm();

    useQuery({
        queryKey: ['notices'],
        queryFn: async () => {
            const response = await getNotice();
            return response.data;
        },
        onSuccess(data) {
            if (data.length > 0) {
                noticeForm.setFieldsValue({
                    notices: data[0].jsonValue.map((notice: any) => {
                        return {
                            description: notice.description,
                            image: notice.image,
                        };
                    }),
                });
            }
        },
    });

    const updateNoticeMutation = useMutation({
        mutationFn: async (body: any) => {
            const response = await updateNotice(body);
            return response.data;
        },
    });

    const updateNoticeHandler = () => {
        noticeForm.validateFields().then((values) => {
            toast.promise(updateNoticeMutation.mutateAsync(values), {
                pending: t('messages:updating-notices'),
                success: t('messages:notices-updated'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    return (
        <Form form={noticeForm} layout="vertical">
            <Form.Item>
                <Form.List name="notices">
                    {(fields, { add, remove, move }) => (
                        <div style={{ display: 'flex', rowGap: 16, flexDirection: 'column' }}>
                            {fields.map((field) => {
                                return (
                                    <div key={field.key} className="w-full flex flex-row gap-4">
                                        {fields.length > 1 && (
                                            <div className="flex flex-col justify-center gap-2">
                                                {field.name === 0 ? (
                                                    <Button
                                                        onClick={() => {
                                                            move(field.name, field.name + 1);
                                                        }}
                                                    >
                                                        <ArrowDownOutlined />
                                                    </Button>
                                                ) : field.name === fields.length - 1 ? (
                                                    <Button
                                                        onClick={() => {
                                                            move(field.name, field.name - 1);
                                                        }}
                                                    >
                                                        <ArrowUpOutlined />
                                                    </Button>
                                                ) : (
                                                    <>
                                                        <Button
                                                            onClick={() => {
                                                                move(field.name, field.name - 1);
                                                            }}
                                                        >
                                                            <ArrowUpOutlined />
                                                        </Button>
                                                        <Button
                                                            onClick={() => {
                                                                move(field.name, field.name + 1);
                                                            }}
                                                        >
                                                            <ArrowDownOutlined />
                                                        </Button>
                                                    </>
                                                )}
                                            </div>
                                        )}
                                        <Card
                                            size="small"
                                            title={`#${field.name + 1}`}
                                            key={field.key}
                                            extra={
                                                <DeleteOutlined
                                                    onClick={() => {
                                                        remove(field.name);
                                                    }}
                                                    className="text-red-500"
                                                />
                                            }
                                            className="w-full"
                                        >
                                            <Form.Item
                                                label={t('description')}
                                                name={[field.name, 'description']}
                                                trigger="onEditorChange"
                                                rules={[{ required: true, message: t('messages:required') }]}
                                            >
                                                <Editor
                                                    id={uuid()}
                                                    tinymceScriptSrc={'/tinymce/js/tinymce/tinymce.min.js'}
                                                    init={{
                                                        height: 500,
                                                        menubar: '',
                                                        plugins: [
                                                            'advlist',
                                                            'autolink',
                                                            'lists',
                                                            'link',
                                                            'searchreplace',
                                                            'code',
                                                            'preview',
                                                            'wordcount',
                                                            'table',
                                                        ],
                                                        toolbar:
                                                            'undo redo | blocks | ' +
                                                            'bold italic forecolor | link | alignleft aligncenter ' +
                                                            'alignright alignjustify | bullist numlist outdent indent | ' +
                                                            'fullscreen',
                                                        block_formats: 'Paragraph=p;Header 2=h2;Header 3=h3;Header 4=h4',
                                                        content_style: 'body { font-family:var(--font-ubuntu); font-size:14px; text-align:left }',
                                                        language: currentLocale === 'en-GB' ? undefined : currentLocale,
                                                        language_url:
                                                            currentLocale === 'en-GB' ? undefined : '/editor-translation/' + currentLocale + '.js',
                                                        promotion: false,
                                                        paste_data_images: true,
                                                        contextmenu: false,
                                                    }}
                                                />
                                            </Form.Item>
                                            <Form.Item label={t('image')} name={[field.name, 'image']}>
                                                <Dragger />
                                            </Form.Item>
                                        </Card>
                                    </div>
                                );
                            })}
                            <Button type="dashed" onClick={() => add()} block>
                                {t('common:add')}
                            </Button>
                        </div>
                    )}
                </Form.List>
            </Form.Item>
            <div className="flex justify-end">
                <Button type="primary" onClick={updateNoticeHandler}>
                    {t('common:update')}
                </Button>
            </div>
        </Form>
    );
};

export default NoticeTab;
