import { useStaffContext } from '@/providers/StaffContext';
import { getCompanyInformation, updateCompanyInformation } from '@/services/site-setting';
import { AxiosErrorResponse, CompanyInformation } from '@/types';
import { errorMessageFormatter } from '@/utils';
import { useMutation, useQuery } from '@tanstack/react-query';
import { Editor } from '@tinymce/tinymce-react';
import { Button, Form, Input } from 'antd';
import { useTranslation } from 'next-i18next';
import { toast } from 'react-toastify';
import { v4 as uuid } from 'uuid';

const CompanyInformationSettingTab: React.FC = () => {
    const { t, i18n } = useTranslation(['site-setting', 'layout', 'common', 'messages']);
    const currentLocale = i18n.language;
    const [companyInformationForm] = Form.useForm();
    const { staff } = useStaffContext();

    // Query
    useQuery({
        queryKey: ['company-information'],
        queryFn: async () => {
            const response = await getCompanyInformation();
            return response.data;
        },
        onSuccess(data) {
            companyInformationForm.setFieldsValue({
                address: data.address,
                email: data.email,
                contactNumber: data.contactNumber,
                bankName: data.bankName,
                accountNumber: data.accountNumber,
                accountHolderName: data.accountHolderName,
                termsAndCondition: data.termsAndCondition,
            });
        },
    });

    const saveBankDetailsMutation = useMutation({
        mutationFn: async (body: CompanyInformation) => {
            const response = await updateCompanyInformation(body);
            return response.data;
        },
    });

    const saveBankDetailsHandler = async () => {
        companyInformationForm.validateFields().then(async (values) => {
            toast.promise(saveBankDetailsMutation.mutateAsync(values), {
                pending: t('messages:updating-company-information'),
                success: t('messages:company-information-updated'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    return (
        <Form form={companyInformationForm} layout="vertical">
            <Form.Item label={t('email')} name="email" rules={[{ required: true, message: t('messages:required') }, { type: 'email' }]}>
                <Input />
            </Form.Item>
            <Form.Item label={t('contact-number')} name="contactNumber" rules={[{ required: true, message: t('messages:required') }]}>
                <Input />
            </Form.Item>
            <Form.Item label={t('bank-name')} name="bankName" rules={[{ required: true, message: t('messages:required') }]}>
                <Input />
            </Form.Item>
            <Form.Item label={t('account-holder-name')} name="accountHolderName" rules={[{ required: true, message: t('messages:required') }]}>
                <Input />
            </Form.Item>
            <Form.Item label={t('account-number')} name="accountNumber" rules={[{ required: true, message: t('messages:required') }]}>
                <Input />
            </Form.Item>
            <Form.Item
                label={t('terms-and-condition')}
                name="termsAndCondition"
                trigger="onEditorChange"
                rules={[{ required: true, message: t('messages:required') }]}
            >
                <Editor
                    id={uuid()}
                    tinymceScriptSrc={'/tinymce/js/tinymce/tinymce.min.js'}
                    init={{
                        height: 500,
                        menubar: '',
                        plugins: ['advlist', 'autolink', 'lists', 'link', 'searchreplace', 'code', 'preview', 'wordcount', 'table'],
                        toolbar:
                            'undo redo | blocks | ' +
                            'bold italic forecolor | link | alignleft aligncenter ' +
                            'alignright alignjustify | bullist numlist outdent indent | ' +
                            'fullscreen',
                        block_formats: 'Paragraph=p;Header 2=h2;Header 3=h3;Header 4=h4',
                        content_style: 'body { font-family:var(--font-ubuntu); font-size:14px; text-align:left }',
                        language: currentLocale === 'en-GB' ? undefined : currentLocale,
                        language_url: currentLocale === 'en-GB' ? undefined : '/editor-translation/' + currentLocale + '.js',
                        promotion: false,
                        paste_data_images: true,
                        contextmenu: false,
                    }}
                />
            </Form.Item>

            {staff?.role.SITE_SETTING && (
                <div className="flex justify-end">
                    <Button loading={saveBankDetailsMutation.isLoading} onClick={saveBankDetailsHandler} type="primary">
                        {t('common:save')}
                    </Button>
                </div>
            )}
        </Form>
    );
};

export default CompanyInformationSettingTab;
