import { Button, Form, MenuProps, Space, Table, Tag } from 'antd';
import { PlusCircleOutlined } from '@ant-design/icons';
import { Trans, useTranslation } from 'next-i18next';
import { useStaffContext } from '@/providers/StaffContext';
import { useState } from 'react';
import { useMutation, useQuery } from '@tanstack/react-query';
import usePagination from '@/hooks/usePagination';
import { Package } from '@/types/package';
import { deletePackage, getPackageList } from '@/services/site-setting';
import FilterDrawer from '../modals/FilterDrawer';
import HtmlParser from '@/components/HtmlParser';
import { ColumnProps } from 'antd/es/table';
import { currencyFormatter, errorMessageFormatter } from '@/utils';
import ActionDropdown from '@/components/ui/ActionDropdown';
import PackageModal from '../modals/PackageModal';
import ConfirmationModal from '@/components/modals/ConfirmationModal';
import { toast } from 'react-toastify';
import { AxiosErrorResponse } from '@/types';

const BookTokenTab: React.FC = () => {
    const { t } = useTranslation(['site-setting', 'common', 'auth']);
    const [pagination, setPagination, paginationOnChange] = usePagination<Package>();
    const { staff } = useStaffContext();
    const [filterPackageForm] = Form.useForm();
    const [isPackageModalOpen, setIsPackageModalOpen] = useState(false);
    const [selectedPackage, setSelectedPackage] = useState<Package | undefined>(undefined);
    const [deletePackageId, setDeletePackageId] = useState<string | undefined>(undefined);

    // Query
    const { data, isFetching, refetch } = useQuery({
        queryKey: ['site-settings', 'packages', 'pagination', pagination],
        queryFn: async () => {
            // Get the form values
            let searchedValue = filterPackageForm.getFieldsValue();

            const query = {
                ...searchedValue,
                page: pagination.page,
                pageSize: pagination.pageSize,
                sortField: pagination.sortField,
                sortOrder: pagination.sortOrder,
            };

            const res = await getPackageList(query);

            setPagination((prevValue) => {
                return {
                    ...prevValue,
                    page: res.data.page,
                    total: res.data.total,
                };
            });

            return res.data.rows;
        },
    });

    const deletePackageMutation = useMutation({
        mutationFn: async () => {
            const res = await deletePackage(deletePackageId!);
            return res.data;
        },
        onSuccess: () => {
            refetch();
        },
    });

    const onResetHandler = () => {
        filterPackageForm.resetFields();
        refetch();
    };

    const onDeleteHandler = () => {
        toast.promise(deletePackageMutation.mutateAsync(), {
            pending: t('messages:deleting-package'),
            success: t('messages:package-deleted'),
            error: {
                render({ data }) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    const columns = [
        // Name
        {
            dataIndex: 'name',
            title: t('name'),
            width: 150,
        },
        // Description
        {
            dataIndex: 'description',
            title: t('description'),
            render: (description: string) => {
                return description ? <HtmlParser html={description} /> : '-';
            },
            width: 300,
        },
        // Status
        {
            dataIndex: 'status',
            title: t('status'),
            render: (status: string) => {
                const statusColor = status === 'ACTIVE' ? 'green' : 'red';
                const statusText = status === 'ACTIVE' ? t('common:active') : t('common:inactive');

                return <Tag color={statusColor}>{statusText}</Tag>;
            },
            width: 100,
        },
        // Number of Token
        {
            dataIndex: 'numberOfBookToken',
            title: t('number-of-book-token'),
            width: 100,
        },
        // Price
        {
            dataIndex: 'price',
            title: t('price'),
            render: (price: number) => {
                return `MYR ${currencyFormatter(price)}`;
            },
            width: 120,
        },
        // Actions
        {
            dataIndex: 'id',
            width: 64,
            align: 'center',
            fixed: 'right',
            render: (_: unknown, record) => {
                const items: MenuProps['items'] = [
                    {
                        label: t('common:edit'),
                        key: 'edit',
                        className: 'text-center',
                        onClick: () => {
                            setIsPackageModalOpen(true);
                            setSelectedPackage(record);
                        },
                    },
                    {
                        label: (
                            <ConfirmationModal
                                title={t('common:delete')}
                                message={<Trans i18nKey={'messages:are-you-sure-you-want-to-delete-this-package'} />}
                                okText={t('common:delete')}
                                onOk={() => {
                                    setDeletePackageId(record.id);
                                    onDeleteHandler();
                                }}
                                reason={false}
                                okButtonProps={{
                                    danger: true,
                                }}
                                width={400}
                            >
                                <Button
                                    loading={deletePackageMutation.isLoading}
                                    type="text"
                                    block
                                    className="!p-0 hover:!text-white"
                                    style={{ transition: 'none' }}
                                >
                                    {t('common:delete')}
                                </Button>
                            </ConfirmationModal>
                        ),
                        key: 'delete',
                        danger: true,
                        className: 'text-center !p-0',
                    },
                ];
                return (
                    <Space>
                        <ActionDropdown items={items} />
                    </Space>
                );
            },
        },
    ] as ColumnProps<Package>[];

    return (
        <div>
            <div className="flex justify-between mb-4 flex-col sm:flex-row">
                {/* Header */}
                <div className="flex">
                    {/* Filter Drawer */}
                    <FilterDrawer filterPackageForm={filterPackageForm} onReset={onResetHandler} onSearch={refetch} loading={isFetching} />
                    {/* Reset Filter Button */}
                    <Button type="link" className="list-none" onClick={onResetHandler}>
                        {t('common:reset-filter')}
                    </Button>
                </div>
                <div className="flex justify-end mb-4">
                    {/* Add Package Button */}
                    {staff?.role.SITE_SETTING && (
                        <Button type="primary" icon={<PlusCircleOutlined />} onClick={() => setIsPackageModalOpen(true)}>
                            {t('add-package')}
                        </Button>
                    )}
                </div>
            </div>
            <div>
                {/* Table */}
                <Table
                    columns={columns}
                    dataSource={data}
                    loading={isFetching}
                    rowKey={(record) => record.id}
                    scroll={{ x: 1000 }}
                    onChange={paginationOnChange}
                    pagination={{
                        current: pagination.page,
                        pageSize: pagination.pageSize,
                        defaultPageSize: 1,
                        showSizeChanger: true,
                        pageSizeOptions: [10, 25, 50, 100],
                        showTotal: (total, range) => t('common:pagination', { range0: range[0], range1: range[1], total: total }),
                        total: pagination.total,
                    }}
                />
            </div>
            <PackageModal
                open={isPackageModalOpen}
                setOpen={setIsPackageModalOpen}
                selectedPackage={selectedPackage}
                setSelectedPackage={setSelectedPackage}
            />
        </div>
    );
};

export default BookTokenTab;
