import { useState } from 'react';
import { useRouter } from 'next/router';
import { Menu as AntMenu, Badge } from 'antd';
import { useTranslation } from 'next-i18next';
import { BookOutlined, FileSearchOutlined, PieChartOutlined, UserOutlined } from '@ant-design/icons';
import { TbHierarchy3 } from 'react-icons/tb';
import { IoBookOutline, IoSettingsOutline } from 'react-icons/io5';
import { CgProfile } from 'react-icons/cg';
import { MdOutlineCalendarMonth, MdOutlineMedicalServices } from 'react-icons/md';
import { IoReceiptOutline } from 'react-icons/io5';
import { HiOutlineUserGroup } from 'react-icons/hi';
import { useQuery } from '@tanstack/react-query';
import { AxiosErrorResponse } from '@/types';
import { errorMessageFormatter } from '@/utils';
import { toast } from 'react-toastify';
import { getAllMedicalCertificate, getAllStudyRequest, getBookTokenRequestCount } from '@/services/data';
import { getAllBookOrder } from '@/services/book-order';

interface MenuProps {
    activeMenu: string[];
    activeDropdown: string[];
}

export const Menu: React.FC<MenuProps> = ({ activeMenu, activeDropdown }) => {
    const { t } = useTranslation(['layout']);
    const router = useRouter();
    const [selectedKeys, setSelectedKeys] = useState<string[]>(activeMenu);
    const [openKeys, setOpenKeys] = useState<string[]>(activeDropdown);

    // Query
    const { data: studyRequest } = useQuery({
        queryKey: ['study-request', 'all'],
        queryFn: async () => {
            const response = await getAllStudyRequest();
            return response.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    const { data: medicalCertificate } = useQuery({
        queryKey: ['medical-certificate', 'all'],
        queryFn: async () => {
            const response = await getAllMedicalCertificate();
            return response.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    const { data: bookTokenRequestCount } = useQuery({
        queryKey: ['book-token-subscription', 'count'],
        queryFn: async () => {
            const response = await getBookTokenRequestCount();
            return response.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    const { data: bookOrderData } = useQuery({
        queryKey: ['book-order', 'count'],
        queryFn: async () => {
            const response = await getAllBookOrder();
            return response.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // book order count where status is pending or unpaid
    const bookOrderCount = bookOrderData?.filter((order) => order.status === 'PENDING' || order.status === 'UNPAID');

    const onSelectMenuHandler = (menu: string, path: string) => {
        setSelectedKeys([menu]);
        router.push(path);
    };

    const menuItems = [
        {
            key: 'dashboard',
            label: t('dashboard'),
            icon: <PieChartOutlined className="!text-base" />,
            onClick: () => onSelectMenuHandler('dashboard', '/dashboard'),
        },
        {
            key: 'birthday-calendar',
            label: t('birthday-calendar'),
            icon: <MdOutlineCalendarMonth className="!text-base" />,
            onClick: () => onSelectMenuHandler('birthday-calendar', '/birthday-calendar'),
        },
        {
            key: 'users',
            label: t('users'),
            icon: <UserOutlined className="!text-base" />,
            children: [
                {
                    key: 'staff',
                    label: t('staff'),
                    icon: <UserOutlined className="!text-base" />,
                    onClick: () => onSelectMenuHandler('staff', '/staff'),
                },
                {
                    key: 'member',
                    label: t('member'),
                    icon: <UserOutlined className="!text-base" />,
                    onClick: () => onSelectMenuHandler('member', '/member'),
                },
            ],
        },

        {
            key: 'role',
            label: t('role'),
            icon: <TbHierarchy3 className="!text-base" />,
            onClick: () => onSelectMenuHandler('role', '/role'),
        },
        {
            key: 'book',
            label: t('book'),
            icon: <IoBookOutline className="!text-base" />,
            onClick: () => onSelectMenuHandler('book', '/book'),
        },
        {
            key: 'study-group-request',
            label: (
                <>
                    {t('study-group-request')}
                    <Badge className="ml-2" count={studyRequest} size="small" />
                </>
            ),
            icon: <FileSearchOutlined className="!text-base" />,
            onClick: () => onSelectMenuHandler('study-group-request', '/study-group-request'),
        },
        {
            key: 'study-group',
            label: t('study-group'),
            icon: <HiOutlineUserGroup className="!text-base" />,
            onClick: () => onSelectMenuHandler('study-group', '/study-group'),
        },
        {
            key: 'medical-certificate',
            label: (
                <>
                    {t('medical-certificate')}
                    <Badge className="ml-2" count={medicalCertificate} size="small" />
                </>
            ),
            icon: <MdOutlineMedicalServices className="!text-base" />,
            onClick: () => onSelectMenuHandler('medical-certificate', '/medical-certificate'),
        },
        {
            key: 'book-token-request',
            label: (
                <>
                    {t('book-token-request')}
                    <Badge className="ml-2" count={bookTokenRequestCount} size="small" />
                </>
            ),
            icon: <IoReceiptOutline className="!text-base" />,
            onClick: () => onSelectMenuHandler('book-token-request', '/book-token-request'),
        },
        {
            key: 'book-order-history',
            label: (
                <>
                    {t('book-order-history')}
                    <Badge className="ml-2" count={bookOrderCount ? bookOrderCount.length : 0} size="small" />
                </>
            ),
            icon: <BookOutlined className="!text-base" />,
            onClick: () => onSelectMenuHandler('book-order-history', '/book-order-history'),
        },
        {
            key: 'settings',
            label: t('menu.settings'),
            type: 'group',
        },
        {
            key: 'site-setting',
            label: t('site-setting'),
            icon: <IoSettingsOutline className="!text-base" />,
            onClick: () => onSelectMenuHandler('site-setting', '/site-setting'),
        },
        {
            key: 'my-profile',
            label: t('my-profile'),
            icon: <CgProfile className="!text-base" />,
            onClick: () => onSelectMenuHandler('my-profile', '/my-profile'),
        },
    ];

    return <AntMenu theme="light" mode="inline" defaultOpenKeys={openKeys} defaultSelectedKeys={selectedKeys} items={menuItems} />;
};
