import usePagination from '@/hooks/usePagination';
import { getStudyRequestList, rejectStudyRequest } from '@/services/book-request';
import { AxiosErrorResponse } from '@/types';
import { conditionalReturn, errorMessageFormatter } from '@/utils';
import { useMutation, useQuery } from '@tanstack/react-query';
import { Button, Form, Table, Tag } from 'antd';
import { useTranslation } from 'next-i18next';
import { toast } from 'react-toastify';
import FilterDrawer from '../modals/StudyRequestFilter';
import { ColumnsType } from 'antd/es/table';
import { StudyRequest } from '@/types/book-request';
import dayjs from 'dayjs';
import { useState } from 'react';
import ConfirmationModal from '@/components/modals/ConfirmationModal';
import ApproveRequestModal from '../modals/ApproveRequest';
import { useStaffContext } from '@/providers/StaffContext';
import { STANDARD_DATE_FORMAT } from '@/configs';
import ColumnSelector from '@/components/modals/ColumnSelector';
import Link from 'next/link';

interface BookStudyRequestProps {
    bookId: string;
}

const BookStudyRequestTab: React.FC<BookStudyRequestProps> = ({ bookId }) => {
    const { t } = useTranslation(['book-request', 'common', 'messages', 'layout']);
    const [pagination, setPagination, paginationOnChange] = usePagination();
    const [filterStudyRequestForm] = Form.useForm();
    const [memberId, setMemberId] = useState<string>('');
    const [requestId, setRequestId] = useState<string>('');
    const [isApproveRequestModalOpen, setIsApproveRequestModalOpen] = useState<boolean>(false);
    const { staff } = useStaffContext();
    const [selectedColumn, setSelectedColumn] = useState<string[]>(['createdAt']);

    // Query
    const bookRequestListQuery = useQuery({
        queryKey: ['book', 'pagination', pagination],
        queryFn: async () => {
            const filterValues = filterStudyRequestForm.getFieldsValue();
            const query = {
                ...filterValues,
                page: pagination.page,
                pageSize: pagination.pageSize,
                sortField: pagination.sortField,
                sortOrder: pagination.sortOrder,
            };
            const response = await getStudyRequestList(bookId, query);

            setPagination((prev) => {
                return {
                    ...prev,
                    page: response.data?.page,
                    total: response.data?.total,
                };
            });

            return response.data?.rows;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Mutation
    const rejectStudyRequestMutation = useMutation({
        mutationFn: async (reason: string) => {
            const response = await rejectStudyRequest(memberId, requestId, reason);
            return response;
        },
        onSuccess() {
            bookRequestListQuery.refetch();
        },
    });

    // Functions
    const onResetFilterHandler = () => {
        filterStudyRequestForm.resetFields();
        bookRequestListQuery.refetch();
    };

    const openApproveRequestModal = () => {
        setIsApproveRequestModalOpen(true);
    };

    const rejectStudyRequestHandler = (reason: string) => {
        toast.promise(rejectStudyRequestMutation.mutateAsync(reason), {
            pending: t('messages:rejecting-book-request'),
            success: t('messages:book-request-rejected'),
            error: {
                render({ data }) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    // Data
    const columns: ColumnsType<StudyRequest> = [
        {
            title: t('name'),
            dataIndex: 'fullName',
            key: 'fullName',
            width: 140,
            render: (_: unknown, bookRequest) => {
                return <Link href={`/member/${bookRequest.member?.id}`}>{bookRequest.member?.fullName}</Link>;
            },
        },
        ...conditionalReturn(selectedColumn.includes('email'), [
            {
                title: t('email'),
                dataIndex: 'email',
                key: 'email',
                render: (_: unknown, bookRequest: StudyRequest) => {
                    return bookRequest.member?.email;
                },
            },
        ]),
        ...conditionalReturn(selectedColumn.includes('phoneNumber'), [
            {
                title: t('phone-number'),
                dataIndex: 'phoneNumber',
                key: 'phoneNumber',
                render: (_: unknown, bookRequest: StudyRequest) => {
                    return bookRequest.member?.phoneNumber;
                },
            },
        ]),
        ...conditionalReturn(selectedColumn.includes('createdAt'), [
            {
                title: t('common:created-at'),
                dataIndex: 'createdAt',
                key: 'createdAt',
                width: 200,
                render: (_: unknown, bookRequest: StudyRequest) => {
                    return dayjs.tz(bookRequest?.createdAt).format(STANDARD_DATE_FORMAT);
                },
            },
        ]),
        {
            title: t('status'),
            dataIndex: 'status',
            key: 'status',
            width: 130,
            render: (_: unknown, bookRequest) => {
                let tagColor, tagText;

                switch (bookRequest?.status) {
                    case 'PENDING':
                        tagColor = 'yellow';
                        tagText = t('pending');
                        break;
                    case 'JOINED':
                        tagColor = 'green';
                        tagText = t('joined');
                        break;
                    default:
                        tagColor = 'red';
                        tagText = t('rejected');
                }

                return <Tag color={tagColor}>{tagText}</Tag>;
            },
        },
        {
            title: t('action'),
            dataIndex: 'id',
            key: 'id',
            width: 205,
            render: (id: string, bookRequest) =>
                bookRequest?.status === 'PENDING' && staff?.role.STUDY_REQUEST_UPDATE ? (
                    <div>
                        <Button
                            type="primary"
                            className="mr-2"
                            onClick={() => {
                                setMemberId(bookRequest.member.id);
                                openApproveRequestModal();
                            }}
                        >
                            {t('approve')}
                        </Button>
                        <ConfirmationModal
                            title={t('common:reject-confirmation')}
                            message={t('are-you-sure-you-want-to-reject-this-book-request?')}
                            okText={t('common:reject')}
                            onOk={(reason) => {
                                setMemberId(bookRequest.member.id);
                                setRequestId(bookRequest.id);
                                rejectStudyRequestHandler(reason as string);
                            }}
                            reason={true}
                            okButtonProps={{
                                danger: true,
                            }}
                        >
                            <Button loading={rejectStudyRequestMutation.isLoading} danger>
                                {t('reject')}
                            </Button>
                        </ConfirmationModal>
                    </div>
                ) : (
                    '-'
                ),
        },
    ];

    const columnOptions = [
        {
            label: t('email'),
            value: 'email',
        },
        {
            label: t('phone-number'),
            value: 'phoneNumber',
        },
        {
            label: t('common:created-at'),
            value: 'createdAt',
        },
    ];

    return (
        <div>
            <div className="flex mb-4"></div>
            <div className="table_container">
                {/* Column Selector */}
                <div className="flex justify-between config_container mb-4">
                    <div className="flex flex-row">
                        <FilterDrawer
                            filterStudyRequestForm={filterStudyRequestForm}
                            onSearch={() => bookRequestListQuery.refetch()}
                            onReset={onResetFilterHandler}
                            loading={bookRequestListQuery.isLoading}
                        />

                        <Button type="link" className="list-none" onClick={onResetFilterHandler}>
                            {t('common:reset-filter')}
                        </Button>
                    </div>
                    <ColumnSelector options={columnOptions} column={selectedColumn} setColumn={setSelectedColumn} />
                </div>
                <Table
                    columns={columns}
                    dataSource={bookRequestListQuery.data}
                    loading={bookRequestListQuery.isLoading}
                    rowKey={(record) => record.id}
                    scroll={{ x: 1000 }}
                    onChange={paginationOnChange}
                    pagination={{
                        current: pagination.page,
                        pageSize: pagination.pageSize,
                        defaultPageSize: 1,
                        showSizeChanger: true,
                        pageSizeOptions: [10, 25, 50, 100],
                        showTotal: (total, range) => t('common:pagination', { range0: range[0], range1: range[1], total: total }),
                        total: pagination.total,
                    }}
                />
            </div>
            <ApproveRequestModal
                open={isApproveRequestModalOpen}
                setOpen={setIsApproveRequestModalOpen}
                refetch={() => bookRequestListQuery.refetch()}
                memberId={memberId}
                bookId={bookId}
            />
        </div>
    );
};

export default BookStudyRequestTab;
