import { AxiosErrorResponse, Book, BookStatus, UploadType } from '@/types';
import { Button, Card, Descriptions, Form, Spin, UploadFile } from 'antd';
import PreviewBook from '../PreviewBook';
import { Trans, useTranslation } from 'next-i18next';
import BookFormFields from '../BookFormFields';
import { useRouter } from 'next/router';
import { useMutation, useQuery, useQueryClient } from '@tanstack/react-query';
import { bookKeys, getBookById, updateBookById, updateBookStatus } from '@/services/book';
import { conditionalReturn, errorMessageFormatter } from '@/utils';
import { toast } from 'react-toastify';
import { mediaFormValueToResponse, mediaResponseToFormValue } from '@/transformer/media/media';
import Initialize from '@/components/globals/Initialize';
import DeleteBookAction from '../DeleteBookAction';
import { useStaffContext } from '@/providers/StaffContext';
import ConfirmationModal from '@/components/modals/ConfirmationModal';
import { dateTransformer } from '@/utils/timeTransformer';
const { Item } = Descriptions;

const BookDetailTab: React.FC = () => {
    const { t } = useTranslation(['book', 'layout', 'common', 'messages']);
    const router = useRouter();
    const [editBookForm] = Form.useForm<Book & { uploadFile: UploadFile[] } & { bookImages: UploadFile[] }>();
    const { bookId, edit } = router.query;
    const queryClient = useQueryClient();
    const { staff } = useStaffContext();

    // Query
    const getBookDetailsQuery = useQuery({
        queryKey: bookKeys.id(bookId as string),
        queryFn: async () => {
            const response = await getBookById(bookId as string);
            return response.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
        onSuccess(data) {
            editBookForm.setFieldsValue({
                name: data.name,
                description: data.description,
                price: data.price,
                availableForBuy: data.availableForBuy,
                remarks: data.remarks,
                uploadType: data.uploadType,
                ...(data.uploadType === UploadType.LINK && { link: data.link }),
                ...conditionalReturn(data.uploadType === UploadType.FILE, { uploadFile: data.bookMedias.map(mediaResponseToFormValue) }),
                bookImages: data.bookImages.map(mediaResponseToFormValue),
            });
        },
    });

    const bookData = getBookDetailsQuery.data as Book;

    // Mutation
    const updateBookMutation = useMutation({
        mutationFn: async (book: Book & { uploadFile: UploadFile[]; bookImages: UploadFile[] }) => {
            const { uploadFile, bookImages, uploadType, ...body } = book;

            let formattedData = null;

            if (uploadType === UploadType.FILE) {
                formattedData = uploadFile.map(mediaFormValueToResponse);
            }

            const formattedImages = bookImages.map(mediaFormValueToResponse);

            const response = await updateBookById(bookId as string, {
                ...body,
                ...conditionalReturn(uploadType === UploadType.FILE, { uploadFile: formattedData }),
                uploadType,
                bookImages: formattedImages,
            });

            return response;
        },
        onSuccess: () => {
            queryClient.invalidateQueries(bookKeys.id(bookId as string));
        },
    });

    const updateBookStatusMutation = useMutation({
        mutationFn: async (bookId: string) => {
            const res = await updateBookStatus(bookId);
            return res.data?.rows;
        },
        onSuccess: () => {
            queryClient.invalidateQueries(bookKeys.id(bookId as string));
        },
    });

    // Function
    const onEditHandler = () => {
        router.push(`/book/${bookId}?edit=true`);
    };

    const onCancelHandler = () => {
        if (edit) {
            return router.push(`/book/${bookId}`);
        }
        router.push(`/book`);
    };

    const onSaveHandler = () => {
        editBookForm.validateFields().then(async (values) => {
            toast.promise(updateBookMutation.mutateAsync(values), {
                pending: t('messages:updating-book'),
                success: t('messages:book-updated'),
                error: {
                    render({ data }) {
                        return t(errorMessageFormatter(data as AxiosErrorResponse));
                    },
                },
            });
        });
    };

    const onUpdateBookStatusHandler = (staffId: string) => {
        toast.promise(updateBookStatusMutation.mutateAsync(staffId), {
            pending: t('messages:updating-book-status'),
            success: {
                render() {
                    queryClient.invalidateQueries([bookKeys.id(bookId as string), 'activityLog']);
                    return t('messages:book-status-updated');
                },
            },
            error: {
                render({ data }: any) {
                    return t(errorMessageFormatter(data));
                },
            },
        });
    };

    if (getBookDetailsQuery.isLoading) {
        return <Initialize />;
    }

    return (
        <Spin spinning={getBookDetailsQuery.isLoading}>
            {getBookDetailsQuery.data && (
                <div className="flex flex-col lg:flex-row gap-8">
                    <div className="flex-1">
                        <Card>
                            {!edit ? (
                                <PreviewBook book={getBookDetailsQuery.data} />
                            ) : (
                                <Form form={editBookForm} layout="vertical">
                                    <BookFormFields isEditMode={true} />
                                </Form>
                            )}
                        </Card>
                    </div>
                    <div className="lg:basis-1/4">
                        <Card>
                            <Descriptions layout="vertical" size="small" column={1}>
                                <Item contentStyle={{ marginBottom: '15px' }} label={t('common:created-at')}>
                                    {dateTransformer(bookData.createdAt)}
                                </Item>
                                <Item contentStyle={{ marginBottom: '15px' }} label={t('actions')}>
                                    <div className="flex flex-col w-full gap-2">
                                        <ConfirmationModal
                                            reason={false}
                                            message={
                                                <Trans
                                                    i18nKey={'messages:are-you-sure-you-want-to-activate-or-deactivate'}
                                                    components={{ strong: <strong /> }}
                                                    values={{
                                                        name: bookData.name,
                                                        action: bookData.status === BookStatus.ACTIVE ? t('common:deactivate') : t('common:activate'),
                                                    }}
                                                />
                                            }
                                            okText={bookData.status === BookStatus.ACTIVE ? t('common:deactivate') : t('common:activate')}
                                            okButtonProps={{
                                                danger: bookData.status === BookStatus.ACTIVE,
                                            }}
                                            onOk={() => {
                                                onUpdateBookStatusHandler(bookData.id);
                                            }}
                                        >
                                            <Button
                                                loading={updateBookStatusMutation.isLoading}
                                                block
                                                type={bookData.status === BookStatus.ACTIVE ? 'default' : 'primary'}
                                                danger={bookData.status === BookStatus.ACTIVE}
                                            >
                                                {bookData.status === BookStatus.ACTIVE ? t('common:deactivate') : t('common:activate')}
                                            </Button>
                                        </ConfirmationModal>
                                        <Button loading={updateBookStatusMutation.isLoading} block onClick={onCancelHandler}>
                                            {t('common:cancel')}
                                        </Button>
                                        {!edit && staff?.role.BOOK_DELETE && <DeleteBookAction book={getBookDetailsQuery.data} />}
                                        {staff?.role.BOOK_UPDATE && (
                                            <Button
                                                loading={updateBookMutation.isLoading}
                                                block
                                                type="primary"
                                                onClick={!edit ? onEditHandler : onSaveHandler}
                                            >
                                                {!edit ? t('common:edit') : t('common:save')}
                                            </Button>
                                        )}
                                    </div>
                                </Item>
                            </Descriptions>
                        </Card>
                    </div>
                </div>
            )}
        </Spin>
    );
};

export default BookDetailTab;
