import { BOOK_FILE_SIZE_LIMIT } from '@/configs';
import { uploadBookFile } from '@/services/book';
import { Book, UploadType } from '@/types';
import { currencyParser, normFile } from '@/utils';
import { PlusOutlined, UploadOutlined } from '@ant-design/icons';
import { Button, Col, Form, Input, InputNumber, Row, Select, Switch, Upload } from 'antd';
import { useTranslation } from 'next-i18next';
import { useState } from 'react';
import { toast } from 'react-toastify';
import PreviewImage from '../globals/PreviewImage';
import usePreviewImage from '@/hooks/usePreviewImage';
import { Editor } from '@tinymce/tinymce-react';
import { v4 as uuid } from 'uuid';

type RadioUploadType = UploadType.LINK | UploadType.FILE;
const BookFormFields = ({ isEditMode }: { isEditMode?: boolean }) => {
    const { t, i18n } = useTranslation(['book', 'layout', 'common', 'messages']);
    const currentLocale = i18n.language;
    const formInstance = Form.useFormInstance<Book>();
    const uploadType = Form.useWatch<RadioUploadType>('uploadType', formInstance);
    const { previewOpen, previewImage, previewTitle, setPreviewOpen, handlePreview } = usePreviewImage();

    const [price, setPrice] = useState<number>(0);

    return (
        <>
            <Row gutter={[16, 0]} className="mb-2">
                <Col span={24}>
                    <Form.Item name="name" label={t('name')} rules={[{ required: true, message: t('messages:required') }]}>
                        <Input size="large" />
                    </Form.Item>
                </Col>
                <Col span={24}>
                    <Form.Item name="description" label={t('description')} trigger="onEditorChange">
                        <Editor
                            id={uuid()}
                            tinymceScriptSrc={'/tinymce/js/tinymce/tinymce.min.js'}
                            init={{
                                height: 300,
                                menubar: '',
                                plugins: ['advlist', 'autolink', 'lists', 'link', 'searchreplace', 'code', 'preview', 'wordcount', 'table'],
                                toolbar:
                                    'undo redo | blocks | ' +
                                    'bold italic forecolor | link | alignleft aligncenter ' +
                                    'alignright alignjustify | bullist numlist outdent indent | ' +
                                    'fullscreen',
                                block_formats: 'Paragraph=p;Header 2=h2;Header 3=h3;Header 4=h4',
                                content_style: 'body { font-family:var(--font-ubuntu); font-size:14px; text-align:left }',
                                language: currentLocale === 'en-GB' ? undefined : currentLocale,
                                language_url: currentLocale === 'en-GB' ? undefined : '/editor-translation/' + currentLocale + '.js',
                                promotion: false,
                                paste_data_images: true,
                                contextmenu: false,
                            }}
                        />
                    </Form.Item>
                </Col>
                <Col>
                    <Form.Item
                        name="bookImages"
                        initialValue={[]}
                        valuePropName="fileList"
                        getValueFromEvent={normFile}
                        label={t('book-images')}
                        rules={[{ required: true, message: t('messages:required') }]}
                    >
                        <Upload
                            customRequest={(options) => uploadBookFile('/staff/book/upload', options)}
                            listType="picture-card"
                            multiple
                            maxCount={15}
                            onPreview={handlePreview}
                            beforeUpload={(file) => {
                                const isLTSizeLimit = file.size / 1024 / 1024 > BOOK_FILE_SIZE_LIMIT;

                                if (isLTSizeLimit) {
                                    toast.error(t('messages:file-too-large', { fileName: file.name, size: BOOK_FILE_SIZE_LIMIT }));
                                    return Upload.LIST_IGNORE;
                                }

                                return true;
                            }}
                            accept="image/png, image/jpeg, image/jpg, .heic, .heif"
                        >
                            <div>
                                <PlusOutlined />
                                <div style={{ marginTop: 8 }}>{t('upload')}</div>
                            </div>
                        </Upload>
                    </Form.Item>
                </Col>
                {/* Available For Buy */}
                {isEditMode && (
                    <Col span={24}>
                        <Form.Item
                            name="availableForBuy"
                            label={t('availableForPurchase')}
                            valuePropName="checked"
                            rules={[{ required: true, message: t('messages:required') }]}
                        >
                            <Switch />
                        </Form.Item>
                    </Col>
                )}
                {/* Price */}
                <Col span={24}>
                    <Form.Item name="price" label={t('price')} rules={[{ required: true, message: t('messages:required') }]}>
                        <InputNumber
                            value={price}
                            onChange={(value) => setPrice(value as number)}
                            step={0.01}
                            min={0.0}
                            className="w-full"
                            prefix={'MYR'}
                            precision={2}
                            parser={currencyParser}
                        />
                    </Form.Item>
                </Col>
                <Col span={24}>
                    <Form.Item name="remarks" label={t('remarks')} trigger="onEditorChange">
                        <Editor
                            id={uuid()}
                            tinymceScriptSrc={'/tinymce/js/tinymce/tinymce.min.js'}
                            init={{
                                height: 300,
                                menubar: '',
                                plugins: ['advlist', 'autolink', 'lists', 'link', 'searchreplace', 'code', 'preview', 'wordcount', 'table'],
                                toolbar:
                                    'undo redo | blocks | ' +
                                    'bold italic forecolor | link | alignleft aligncenter ' +
                                    'alignright alignjustify | bullist numlist outdent indent | ' +
                                    'fullscreen',
                                block_formats: 'Paragraph=p;Header 2=h2;Header 3=h3;Header 4=h4',
                                content_style: 'body { font-family:var(--font-ubuntu); font-size:14px; text-align:left }',
                                language: currentLocale === 'en-GB' ? undefined : currentLocale,
                                language_url: currentLocale === 'en-GB' ? undefined : '/editor-translation/' + currentLocale + '.js',
                                promotion: false,
                                paste_data_images: true,
                                contextmenu: false,
                            }}
                        />
                    </Form.Item>
                </Col>
                <Col span={24}>
                    <Form.Item name="uploadType" label={t('upload-type')}>
                        <Select size="large" placeholder={t('common:select-if-applicable')}>
                            <Select.Option value={UploadType.NONE}>{t('none')}</Select.Option>
                            <Select.Option value={UploadType.LINK}>{t('link')}</Select.Option>
                            <Select.Option value={UploadType.FILE}>{t('file')}</Select.Option>
                        </Select>
                    </Form.Item>
                </Col>
                {uploadType && (
                    <Col span={24}>
                        {uploadType === UploadType.LINK ? (
                            <Form.Item name="link" label={t('link')} rules={[{ required: true, message: t('messages:required') }]}>
                                <Input className="!w-full" size="large" placeholder="https://example.com" />
                            </Form.Item>
                        ) : uploadType === UploadType.FILE ? (
                            <Form.Item
                                name="uploadFile"
                                valuePropName="fileList"
                                getValueFromEvent={normFile}
                                label={t('upload-file')}
                                rules={[{ required: true, message: t('messages:required') }]}
                            >
                                <Upload
                                    multiple
                                    listType="picture"
                                    className="add-book-file-uploader"
                                    beforeUpload={(file) => {
                                        const isLTSizeLimit = file.size / 1024 / 1024 > BOOK_FILE_SIZE_LIMIT;

                                        if (isLTSizeLimit) {
                                            toast.error(t('messages:file-too-large', { fileName: file.name, size: BOOK_FILE_SIZE_LIMIT }));
                                            return Upload.LIST_IGNORE;
                                        }

                                        return true;
                                    }}
                                    customRequest={(options) => uploadBookFile('/staff/book/upload', options)}
                                >
                                    <Button icon={<UploadOutlined />} block className="!w-full" size="large">
                                        {t('click-to-upload')}
                                    </Button>
                                </Upload>
                            </Form.Item>
                        ) : null}
                    </Col>
                )}
            </Row>
            <PreviewImage open={previewOpen} setOpen={setPreviewOpen} title={previewTitle} src={previewImage} />
        </>
    );
};

export default BookFormFields;
