import { Trans, useTranslation } from 'next-i18next';
import ConfirmationModal from '../modals/ConfirmationModal';
import { AxiosErrorResponse } from '@/types';
import { toast } from 'react-toastify';
import { useMutation } from '@tanstack/react-query';
import { errorMessageFormatter } from '@/utils';
import { Button } from 'antd';
import { BookOrderList } from '@/types/book-order';
import { deleteBookOrderById } from '@/services/book-order';

interface DeleteBookOrderActionProps {
    bookOrder: BookOrderList;
    refetch: () => void;
}

const DeleteBookOrderAction: React.FC<DeleteBookOrderActionProps> = ({ bookOrder, refetch }) => {
    const { t } = useTranslation(['book-order', 'layout', 'common', 'messages']);

    const deleteBookOrderMutation = useMutation({
        mutationFn: async () => {
            const response = await deleteBookOrderById(bookOrder.id);
            return response.data;
        },
        onSuccess: () => {
            refetch();
        },
    });

    const onDeleteHandler = () => {
        toast.promise(deleteBookOrderMutation.mutateAsync(), {
            pending: t('messages:deleting-book-order'),
            success: t('messages:book-order-deleted'),
            error: {
                render({ data }) {
                    return t(errorMessageFormatter(data as AxiosErrorResponse));
                },
            },
        });
    };

    return (
        <ConfirmationModal
            title={t('common:delete-confirmation')}
            message={<Trans i18nKey={'messages:are-you-sure-you-want-to-delete-this-book-order'} components={{ strong: <strong /> }} />}
            okText={t('common:delete')}
            onOk={onDeleteHandler}
            reason={false}
            okButtonProps={{
                danger: true,
            }}
            width={400}
        >
            <Button
                loading={deleteBookOrderMutation.isLoading}
                type="link"
                ghost
                block
                className="!p-0 hover:!text-white text-black"
                style={{ transition: 'none' }}
            >
                {t('common:delete')}
            </Button>
        </ConfirmationModal>
    );
};

export default DeleteBookOrderAction;
