import { useTranslation } from 'next-i18next';
import { Calendar, Spin, Tag } from 'antd';
import { CalendarMode as AntdCalendarMode, CalendarProps, SelectInfo } from 'antd/es/calendar/generateCalendar';
import dayjs, { Dayjs } from 'dayjs';
import { useBirthdayContext } from '@/providers/BirthdayContext';
import { BirthdayDateQuery, CalendarMode } from '@/types';
import React from 'react';

const BirthdayModule: React.FC = () => {
    const { t } = useTranslation(['birthday', 'layout', 'common', 'messages']);
    const { calendar } = useBirthdayContext();
    const { date, setDate, mode, setMode, setBirthdayListModalOpen, birthdayListQuery } = calendar;
    const [month, setMonth] = React.useState<number>(dayjs.tz().month());

    const calendarDataYear = birthdayListQuery.data || [];

    const calendarDataMonth =
        birthdayListQuery.data?.filter((date: any) => {
            return dayjs.tz(date.dateOfBirth).month() === month;
        }) || [];

    const cellRender: CalendarProps<Dayjs>['cellRender'] = (current, info) => {
        const monthlyBirthdays = calendarDataMonth.filter((data: BirthdayDateQuery) => {
            const birthdayDate = dayjs.tz(data.dateOfBirth);
            return birthdayDate.month() === current.month() && birthdayDate.date() === current.date();
        }).length;

        const yearlyBirthdays = calendarDataYear.filter((data: BirthdayDateQuery) => {
            const birthdayDate = dayjs.tz(data.dateOfBirth);
            return birthdayDate.month() === current.month();
        }).length;

        const renderTag = (count: number) => (
            <div>
                {count > 0 && (
                    <div>
                        <Tag color="green">
                            <span>{count}</span> {count > 1 ? t('birthdays') : t('birthday')}
                        </Tag>
                    </div>
                )}
            </div>
        );

        if (mode === CalendarMode.Month) {
            return renderTag(monthlyBirthdays);
        } else {
            return renderTag(yearlyBirthdays);
        }
    };

    const onSelect = (value: Dayjs, info: SelectInfo) => {
        if (info.source === 'date') {
            if (date.isSame(value, 'month')) {
                setBirthdayListModalOpen(true);
            }
            setDate(value);
        }

        if (info.source === 'month') {
            setMode(CalendarMode.Month);
        }
    };

    const onPanelChange = (date: Dayjs, source: AntdCalendarMode) => {
        setMonth(dayjs.tz(date).month());
        setMode(source as CalendarMode);
        setDate(date);
    };

    const onChange = (date: Dayjs) => {
        setDate(date);
    };

    return (
        <div className="overflow-scroll">
            <Spin spinning={birthdayListQuery?.isLoading || birthdayListQuery?.isRefetching}>
                <Calendar
                    cellRender={cellRender}
                    onSelect={onSelect}
                    style={{ minWidth: 700 }}
                    mode={mode}
                    value={date}
                    onChange={onChange}
                    onPanelChange={onPanelChange}
                />
            </Spin>
        </div>
    );
};

export default BirthdayModule;
