import { useEffect, useState } from 'react';
import { useTranslation } from 'next-i18next';
import PasswordHint from './PasswordHint';

interface PasswordHintContainerProps {
    password: string;
}

const PasswordHintContainer: React.FC<PasswordHintContainerProps> = ({ password }) => {
    const { t } = useTranslation(['auth', 'messages']);

    const [isLengthFulfilled, setIsLengthFulfilled] = useState<boolean>(false);
    const [isLowercaseFulfilled, setIsLowercaseFulfilled] = useState<boolean>(false);
    const [isUppercaseFulfilled, setIsUppercaseFulfilled] = useState<boolean>(false);
    const [isSpecialFulfilled, setIsSpecialFulfilled] = useState<boolean>(false);
    const [isNumberFulfilled, setIsNumberFulfilled] = useState<boolean>(false);

    useEffect(() => {
        // Validate password length
        if (password.length < 8 || password.length > 20) {
            setIsLengthFulfilled(false);
        } else {
            setIsLengthFulfilled(true);
        }

        // Validate password lowercase
        if (/[a-z]/.test(password)) {
            setIsLowercaseFulfilled(true);
        } else {
            setIsLowercaseFulfilled(false);
        }

        // Validate password uppercase
        if (/[A-Z]/.test(password)) {
            setIsUppercaseFulfilled(true);
        } else {
            setIsUppercaseFulfilled(false);
        }

        // Validate password special character
        if (/[^A-Za-z0-9]/.test(password)) {
            setIsSpecialFulfilled(true);
        } else {
            setIsSpecialFulfilled(false);
        }

        // Validate password number
        if (/[0-9]/.test(password)) {
            setIsNumberFulfilled(true);
        } else {
            setIsNumberFulfilled(false);
        }
    }, [password]);

    return (
        <div className="mb-2">
            <div>{t('hint.password-must-contains')}:</div>
            <PasswordHint isFulfilled={isLengthFulfilled} passwordHint={t('hint.8-20-characters')} />
            <PasswordHint isFulfilled={isLowercaseFulfilled} passwordHint={t('hint.minimum-1-lowercase-character')} />
            <PasswordHint isFulfilled={isUppercaseFulfilled} passwordHint={t('hint.minimum-1-uppercase-character')} />
            <PasswordHint isFulfilled={isNumberFulfilled} passwordHint={t('hint.minimum-1-number')} />
            <PasswordHint isFulfilled={isSpecialFulfilled} passwordHint={t('hint.minimum-1-special-character')} />
        </div>
    );
};

export default PasswordHintContainer;
