import { useTranslation } from 'next-i18next';
import dayjs from 'dayjs';
import { Card, Divider, Empty, Tag } from 'antd';
import React from 'react';
import { useStudyGroupContext } from '@/providers/StudyGroupContext';
import PreviewTask from '@/components/study-group-task/modals/PreviewTask';
import { useMemberContext } from '@/providers/MemberContext';
import { extractMemberFromTask } from '@/transformer/study-group';
import { dateTransformer } from '@/utils/timeTransformer';
import { useQuery } from '@tanstack/react-query';
import { useRouter } from 'next/router';
import { getMemberStudyGroupComments } from '@/services/study-group';
import { AxiosErrorResponse } from '@/types';
import { errorMessageFormatter } from '@/utils';
import { toast } from 'react-toastify';

const Task: React.FC = () => {
    const { t } = useTranslation(['study-group', 'common', 'messages', 'layout']);
    const { studyGroup, setStudyGroupTaskId } = useStudyGroupContext();
    const [isPreviewTaskModal, setIsPreviewTaskModal] = React.useState<boolean>(false);
    const { member } = useMemberContext();
    const router = useRouter();
    const { studyGroupId } = router.query;

    // Query
    const { data: ownCommentData } = useQuery({
        queryKey: ['study-group-own-comment', studyGroupId],
        enabled: !!studyGroupId,
        queryFn: async () => {
            const res = await getMemberStudyGroupComments(studyGroupId as string);
            return res.data;
        },
        onError: (error: AxiosErrorResponse & Error) => {
            toast.error(t(errorMessageFormatter(error)));
        },
    });

    // Function
    const renderTaskDate = (taskDate: string) => {
        const currentDate = dayjs.tz();
        const parsedTaskDate = dayjs(taskDate).tz('Asia/Kuala_Lumpur');

        if (parsedTaskDate.isBefore(currentDate, 'day')) {
            // Task date is in the past
            return (
                <Tag color="#ff7b5d" className="w-fit">
                    {t('past')}
                </Tag>
            );
        } else if (parsedTaskDate.isSame(currentDate, 'day')) {
            // Task date is today
            return (
                <Tag color="#6ce66c" className="w-fit">
                    {t('today')}
                </Tag>
            );
        } else {
            // Task date is in the future
            return (
                <Tag color="#ffcc5d" className="w-fit">
                    {t('future')}
                </Tag>
            );
        }
    };

    return (
        <div>
            <div className="mt-4">
                <div className="mt-2 grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4">
                    {studyGroup.data?.studyGroupTask &&
                        studyGroup.data.studyGroupTask.map((task) => {
                            const taskDate = dateTransformer(task.taskDate);
                            const isBeforeCurrentDate = dayjs(task.taskDate).tz().isBefore(dayjs.tz(), 'day');
                            const isMemberPreScheduled = ownCommentData?.find(
                                (comment) => comment.prePlanned && comment.studyGroupTaskId === task.id,
                            );
                            const isMemberCommented = extractMemberFromTask(task).includes(member?.id as string);

                            return (
                                <div
                                    key={task.id}
                                    onClick={() => {
                                        setIsPreviewTaskModal(true);
                                        setStudyGroupTaskId(task.id);
                                    }}
                                >
                                    <Card className="cursor-pointer h-fit overflow-y-auto" hoverable>
                                        <p className="text-sm md:text-base">
                                            <div className="flex justify-between ">
                                                <div>{taskDate}</div>
                                                <div>{renderTaskDate(task.taskDate)}</div>
                                            </div>
                                            <Divider />
                                            <div className="flex justify-between items-center">
                                                {isMemberPreScheduled ? (
                                                    <Tag color="blue" className="w-fit">
                                                        {t('pre-scheduled')}
                                                    </Tag>
                                                ) : isMemberCommented ? (
                                                    <Tag color="green" className="w-fit">
                                                        {t('commented')}
                                                    </Tag>
                                                ) : isBeforeCurrentDate ? (
                                                    <Tag color="red" className="w-fit">
                                                        {t('not-commented')}
                                                    </Tag>
                                                ) : null}
                                            </div>
                                        </p>
                                    </Card>
                                </div>
                            );
                        })}
                </div>
                {!studyGroup.data?.studyGroupTask?.length && <Empty description={t('no-task-available')} />}
            </div>
            <PreviewTask open={isPreviewTaskModal} setOpen={setIsPreviewTaskModal} />
        </div>
    );
};

export default Task;
