import { useEffect, useState } from 'react';
import { MdOutlineEdit } from 'react-icons/md';
import { useTranslation } from 'next-i18next';
import { UseQueryResult } from '@tanstack/react-query';
import { Button, Input, Spin, Typography } from 'antd';
import { Member } from '@/types';
import PasswordModal from '../modals/PasswordModal';
import PhoneNumberModal from '../modals/PhoneModal';

interface CredentialTabProps {
    memberId: string;
    memberQuery: UseQueryResult<Member | null>;
}

const Credential: React.FC<CredentialTabProps> = ({ memberQuery }) => {
    const { t } = useTranslation(['member', 'common']);
    const memberToBeEdit = memberQuery.data;
    const { Title } = Typography;

    const [email, setEmail] = useState<string>('');
    const [phoneNumber, setPhoneNumber] = useState<string>('');
    const [isPhoneNumberModalOpen, setIsPhoneNumberModalOpen] = useState<boolean>(false);
    const [isPasswordModalOpen, setIsPasswordModalOpen] = useState<boolean>(false);

    useEffect(() => {
        if (memberToBeEdit) {
            setEmail(memberToBeEdit.email);
            setPhoneNumber(memberToBeEdit.phoneNumber);
        }
    }, [memberToBeEdit]);

    return (
        <Spin spinning={memberQuery.isLoading}>
            {memberToBeEdit && (
                <div>
                    {/* Email Input */}
                    <div className="mb-6">
                        <Title level={5}>{t('email')}</Title>
                        <div className="flex gap-2 items-center">
                            <Input name="email" disabled value={email} className="max-w-[300px]" />
                        </div>
                    </div>
                    {/* Phone Number Input */}
                    <div className="mb-6">
                        <Title level={5}>{t('phone-number')}</Title>
                        <div className="flex gap-2 items-center">
                            <Input name="phoneNumber" disabled value={phoneNumber} className="max-w-[300px]" />
                            <Button
                                type="primary"
                                onClick={() => {
                                    setIsPhoneNumberModalOpen(true);
                                }}
                            >
                                <MdOutlineEdit className="text-xl" />
                            </Button>
                        </div>
                    </div>
                    {/* Password Input */}
                    <div className="mb-4">
                        <Title level={5}>{t('password')}</Title>
                        <div className="flex gap-2 items-center">
                            <Input.Password disabled defaultValue="************" className="max-w-[300px]" visibilityToggle={false} />
                            <Button type="primary" onClick={() => setIsPasswordModalOpen(true)}>
                                <MdOutlineEdit className="text-xl" />
                            </Button>
                        </div>
                    </div>
                    <PhoneNumberModal
                        open={isPhoneNumberModalOpen}
                        setOpen={setIsPhoneNumberModalOpen}
                        member={memberToBeEdit}
                        memberQuery={memberQuery}
                    />
                    <PasswordModal open={isPasswordModalOpen} setOpen={setIsPasswordModalOpen} memberQuery={memberQuery} />
                </div>
            )}
        </Spin>
    );
};

export default Credential;
