import { Book, UploadType } from '@/types';
import { currencyFormatter, mediaUrlGenerator } from '@/utils';
import { Col, Row, Tag } from 'antd';
import { useTranslation } from 'next-i18next';
import HtmlParser from '../HtmlParser';

interface PreviewBookProps {
    book: Book;
}

const PreviewBook: React.FC<PreviewBookProps> = ({ book }) => {
    const { t } = useTranslation(['book', 'layout', 'common', 'messages']);

    return (
        <Row gutter={[16, 16]}>
            <Col span={24}>
                <p className="font-bold">{t('name')}:</p>
                <p>{book.name}</p>
            </Col>
            <Col span={24}>
                <p className="font-bold">{t('description')}:</p>
                <HtmlParser html={book.description ?? '-'} />
            </Col>
            <Col span={24}>
                {book.bookImages && (
                    // Image preview
                    <div>
                        <p className="font-bold">{t('book-images')}:</p>
                        <div className="flex flex-wrap gap-4">
                            {book.bookImages.map((image) => (
                                <img
                                    key={image.media.id}
                                    src={mediaUrlGenerator(image.media.key)}
                                    alt={image.media.name}
                                    className="w-[200px] h-[200px]"
                                />
                            ))}
                        </div>
                    </div>
                )}
            </Col>
            <Col span={24}>
                <p className="font-bold">{t('availableForPurchase')}:</p>
                <div className="mt-1">
                    <span>{book.availableForBuy ? <Tag color="green">{t('yes-purchase')}</Tag> : <Tag>{t('no-purchase')}</Tag>}</span>
                </div>
            </Col>
            <Col span={24}>
                <p className="font-bold">{t('price')}:</p>
                <p>RM {currencyFormatter(book.price)}</p>
            </Col>
            <Col span={24}>
                <p className="font-bold">{t('remarks')}:</p>
                <HtmlParser html={book?.remarks ?? '-'} />
            </Col>
            <Col span={24}>
                {book.uploadType === UploadType.LINK ? (
                    <>
                        <p className="font-bold">{t('link')}:</p>
                        <a target="_blank" href={book.link}>
                            {book.link}
                        </a>
                    </>
                ) : book.uploadType === UploadType.FILE ? (
                    <>
                        <p className="font-bold">{t('file')}:</p>
                        {book.bookMedias.map((media) => (
                            <p key={media.media.id}>
                                <a target="_blank" href={mediaUrlGenerator(media.media.key)}>
                                    {media.media.name}
                                </a>
                            </p>
                        ))}
                    </>
                ) : null}
            </Col>
        </Row>
    );
};

export default PreviewBook;
